package mChaRM.multichannel;

import java.io.*;
import uka.karmi.rmi.*;
import uka.karmi.rmi.server.*;
import java.lang.*;
import java.util.Hashtable;
import java.util.Arrays;
import java.util.Vector;
import java.net.MalformedURLException;
import mChaRM.RMI.*;

  /** 
   * This class defines how multi-channels behave (by default) in the abstract locus. Any other kind 
   * of multi-channel can be derived from this class. The key issue 
   * consists in overriding the {@link #coreMetaBehavior <B>coreMetaBehavior</B>} method.
   * <HR>
   * <B>Remember</B> each multi-channel is identified by its <I>kind</I>, ie., the reflective behaviour 
   * it realizes, and by the set of objects -- playing the role of receivers -- which it is connected to. 
   * @author  Walter Cazzola (<A HREF="mailto:cazzola@disi.unige.it">cazzola@disi.unige.it</A>)
   * @version 1.5
   * @since Version 1.0
   **/

public class channelCore extends UnicastRemoteObject implements channelInterface {
  /**
   * String representing the reflective behavior (<I>kind</I>) realized by multi-channels 
   * instanciated from this class. 
   * <B>Remember</B> each multi-channel is identified by its kind and by the set of
   * objects -- playing the role of receivers -- which it is connected to. 
   * @serial
   **/
  private String kind;
  /**
   * Strings representing the objects which the multi-channel is connected to.
   * <B>Remember</B> each multi-channel is identified by its kind and by the set of
   * objects -- playing the role of receivers -- which it is connected to. 
   * @serial
   **/
  private String[] receiversName;
  /**
   * Hashtable indexed on receivers' name; each entry contains the interface of the stub connected
   * to the receiver. These interfaces are gates used by the core for coordinating the whole multi-channel
   * behavior. 
   * @serial
   **/
  private Hashtable receivers;
  /**
   * Strings representing the objects which contact the multi-channel.
   * @serial
   **/
  private Vector sendersName;
  /**
   * Hashtable indexed on sender names; each entry contains the interface of the stub connected
   * to the sender. These interfaces are gates used by the core for coordinating the whole multi-channel
   * behavior. 
   * @serial
   **/
  private Hashtable senders;
  /**
   * The name of the class this kind of multi-channel uses as receiverStub. This approach allows us 
   * to define multi-channel which redefine the stubs behavior.
   * @serial
   **/
  private String receiverStubClassName;
  /**
   * The name of the class that this kind of multi-channel uses as senderStub. This approach allows us 
   * to define multi-channel which redefine the stubs behavior.
   * @serial
   **/
  private String senderStubClassName;
  /** @serial **/
  private String serverName;
  
                     //******* accessors section *******// 
    /**
     * returns the <B>kind</B> of the multi-channel.
     * <HR>
     * @return the kind of the multi-channel.
     **/
  public String kind() {return kind;}
    /**
     * sets the <B>kind</B> of the multi-channel to the passed value <B>k</B>.
     * <HR>
     * @param k the <I>kind</I> of the multi-channel.
     **/
  final public void setKind(String k) {kind=k;}
    /**
     * returns the name of the receiver whose position is specified by indx. 
     * The position is related to how the names are stored. At the moment this method can be used consistently only in conjunction with loop statements. 
     * <HR>
     * @param indx the index of the position.
     * @return the name of the receiver. 
     **/
  final public String receiverName(int indx) {return receiversName[indx];}
    /**
     * inserts the name of a receiver at the position specified by indx. 
     * <HR>
     * The method doesn't check if the slot is already used.<BR>
     * Note that you have to keep the set of the name alphabetically sorted in order to allow the core to connect to its stubs and vice versa.
     * <HR>
     * @param indx the index of the position.
     * @param name the name of the receiver. 
     **/
  final public void setReceiverName(int indx, String name) {receiversName[indx]=name;}
    /**
     * uses the array of strings specified by name as the set of names of the receivers of the multi-channel. It also sorts the array.
     * <HR>
     * @param names a set of names. 
     **/
  final public void setRsName(String[] names) {
    Arrays.sort(names, String.CASE_INSENSITIVE_ORDER);
    receiversName = names;
  }
    /**
     * returns the number of receivers bound to this multi-channel. 
     * <HR>
     * The number of receivers is a quantities fixed at compile-time and depends on the <B>kinds</B> statement.
     * <HR>
     * @return the number of receivers.
     **/
  final public int howManyReceivers() {return receiversName.length;}
    /**
     * returns the name of the sender whose position is specified by indx. 
     * <HR>
     * The position is related to how the names are stored. At the moment this method can be used consistently only in conjunction with loop statements. 
     * <HR>
     * @param indx the index of the position.
     * @return the name of the sender. 
     * @exception ArrayIndexOutOfBoundsException thrown when the index passed is too big with respect to the number of the senders.
     **/
  final public String senderName(int indx) throws ArrayIndexOutOfBoundsException {return (sendersName.elementAt(indx)).toString();}
    /**
     * inserts the name of a sender at the position specified by indx. 
     * <HR>
     * The method doesn't check if the slot is already used. The method provides to append a new sender to the tail when the current number of sender is passed as index.
     * <HR>
     * @param indx the index of the position.
     * @param name the name of the receiver. 
     **/
  final public void setSenderName(int indx, String name) throws ArrayIndexOutOfBoundsException {
    if (indx == sendersName.size()) sendersName.insertElementAt(name, indx);
    else sendersName.setElementAt(name, indx);
  }
    /**
     * returns the number of senders bound to this multi-channel, at the moment of the call.
     * <HR>
     * The number of senders is zero at the core creation and increase during the execution with the calls the core has to process.
     * <HR>
     * @return the number of senders.
     **/
  final public int howManySenders() {return sendersName.size();}
  final public receiverStubInterface receiverStub(String name) {return (receiverStubInterface)(receivers.get(name));}
  final public receiverStubInterface receiverStubByPosition(int indx) {return (receiverStubInterface)(receivers.get(receiverName(indx)));}
  final public void setReceiverStub(String name, receiverStubInterface RSI){receivers.put(name, RSI);}
  final public senderStubInterface senderStub(String name) {return (senderStubInterface)(senders.get(name));}
  final public senderStubInterface senderStubByPosition(int indx) {return (senderStubInterface)(senders.get(senderName(indx)));}
  final public void setSenderStub(String name, senderStubInterface SSI){senders.put(name, SSI);}
    
  final public String receiverStubClassName() {return receiverStubClassName;}
  final public void setReceiverStubClassName(String name) {receiverStubClassName=name;}
  final public String senderStubClassName() {return senderStubClassName;}
  final public void setSenderStubClassName(String name) {senderStubClassName=name;}
  
    /** 
     * constructor of multi-channels.
     * <HR>
     * This constructor creates and registers as server the core of a multi-channel of <I>kind</I>: <B>kind</B>. 
     * The core is registered using the following string:
     * <CENTER>"__"+<B>kind</B>+"{ "rec<SUB>1</SUB>+" "+rec<SUB>2</SUB> ... +rec<SUB>n</SUB>+" }"</CENTER>
     * then, it instructs each receivers to create and to register as server their stubs.
     * The stub of the receiver <I>rec<SUB>i</SUB></I> is registered as:
     * <CENTER>"__"+<B>kind</B>+"{ "rec<SUB>1</SUB>+" "+rec<SUB>2</SUB> ... +rec<SUB>n</SUB>+" }"+"__"+rec<SUB>i</SUB></CENTER>
     * finally the core connect itself with its stubs on the receivers site.
     * <HR>
     * @param kind a String representing the multi-channel's behavior.
     * @param RsName an Array of Strings representing the name of the receivers, which it is connected to.
     * @param SSClassName a String specifying the class of the stubs that will be attached to the multi-channel senders.
     * @param RSClassName a String specifying the class of the stubs that will be attached to the multi-channel receivers.
     * @exception ReceiverStubNotFoundException thrown when one of the specified receiver wasn't started before the core.
     **/

  public channelCore(String kind, String[] RsName, String SSClassName, String RSClassName) throws RemoteException, ReceiverStubNotFoundException {
    int i;
    receivers = new Hashtable(); senders = new Hashtable();
    setReceiverStubClassName(RSClassName); setSenderStubClassName(SSClassName);
    setRsName(RsName); setKind(kind); sendersName = new Vector(10, 1);
                                       // I'm building up the name which the channel use to register itself as server
    serverName = "__"+kind()+"{ ";
    for (i=0; i<howManyReceivers(); i++) serverName += receiverName(i) + " ";
    serverName += "}";
                                       // I'm registering myself as server
    try {
      Naming.rebind(serverName, this); 
    } catch(MalformedURLException e) {
      System.out.println("*** Troubles in registering the core as a RMI server");
      e.printStackTrace();
    }
                                       // I'm instructing the receivers about creating their stubs
    try {
      reflectiveReceiverInterface RSI;
      for (i=0; i<howManyReceivers(); i++) {
        RSI = (reflectiveReceiverInterface)lookupEveryWhere.lookup(receiverName(i));  
        RSI.attachingStub(serverName, receiverStubClassName(), receiverName(i));
        setReceiverStub(receiverName(i), (receiverStubInterface)lookupEveryWhere.lookup(serverName+"__"+receiverName(i)));
      }
    } catch(Exception e) { 
        throw new ReceiverStubNotFoundException("*** Troubles in detecting remote receiver stubs: "+receiverName(i)+" I look for: "+serverName+"__"+receiverName(i));
    }
  }  

   /** This method embodies the reflective behaviour realized by the multi-channel.
    * <HR>
    * This method performs meta-computations on the reified method call:
    * <CENTER>multiRMI(RsName, methodName, args)</CENTER>
    * the computations performed can't make assumption about where their are performed.<BR>
    * This method <B>must</B> be overrode in order to implement new kind of multi-channels.<BR><BR>
    * as default behavior, it delivers the message to the specified receivers, collects the results 
    * and returns the result of the computation of the first receiver to the sender.
    * <HR>   
    * Note that receivers can be a subset of the multi-channel's receivers
    * <HR>
    * @param msg the hijacked method call.
    * @return the result of the method call.
    * @exception MethodDoesNotExistException thrown when the method reified doesn't exist in the referent class.
    **/
 
  public Object coreMetaBehavior(mChaRMMethodCall msg) throws MethodDoesNotExistException, RemoteException {
    Object[] result = new Object[(msg.receivers()).length];
    for(int i=0; i<(msg.receivers()).length; i++)
      result[i] = ((receiverStubInterface)receiverStub((msg.receivers())[i])).invoke(msg);
    return result[0]; 
  }

   /**
    * retrieves the content of a field of the specified sender.
    * <HR>
    * @param senderName the name of the sender the core questions about the content of its field
    * @param fieldName the name of the field we want to inspect.
    * @return the content of the inspected field.
    * @exception SenderStubNotFoundException thrown when the sender is unreacheable.
    * @exception FieldDoesNotExistException thrown when the class of the sender doesn't define the field passed.
    **/
  public Object retrieveSenderFieldValue(String senderName, String fieldName) throws SenderStubNotFoundException, FieldDoesNotExistException, RemoteException {
    senderStubInterface SSInterface = (senderStubInterface)senderStub(senderName);
    if (SSInterface == null) throw new SenderStubNotFoundException();
    return SSInterface.retrieveField(fieldName);
  }
   /**
    * retrieves the content of a field of the specified receiver.
    * <HR>
    * @param receiverName the name of the receiver the core questions about the content of its field
    * @param fieldName the name of the field we want to inspect.
    * @return the content of the inspected field.
    * @exception ReceiverStubNotFoundException thrown when the receiver is unreacheable.
    * @exception FieldDoesNotExistException thrown when the class of the sender doesn't define the field passed.
    **/
  public Object retrieveReceiverFieldValue(String receiverName, String fieldName) throws ReceiverStubNotFoundException, FieldDoesNotExistException, RemoteException {
    receiverStubInterface RSInterface = (receiverStubInterface)receiverStub(receiverName);
    if (RSInterface == null) throw new ReceiverStubNotFoundException();
    return RSInterface.retrieveField(fieldName);
  }
 
   /** on request, it supplies the class name of its senderStub to the requiring client. 
    * <HR>
    * when a sender needs that one of its computation be reified by a multi-channel 
    * it looks for a sender stub of such multi-channel.
    * <HR>
    * This method cannot be overrode nor used directly by users.
    * <HR>
    * <B>returns</B>: the class name of its senderStub.
    **/
  final public String supplyASenderStub() throws RemoteException {
    return senderStubClassName();
  }
  
   /** the invocation of this method allows to connect the multi-channel to the new stub. 
    * <HR>
    * This method cannot be overrode nor used directly by users.
    * <HR>
    * @param senderStubName the name of the stub, who the multi-channel have to connect to.
    * @exception SenderStubNotFoundException thrown when the senderStub is not ready to be linked to the core. 
    **/
  final public void senderStubHasBeenCreated(String senderStubName) throws SenderStubNotFoundException {
    try {
      setSenderName(howManySenders(), senderStubName);
      setSenderStub(senderStubName, (senderStubInterface)lookupEveryWhere.lookup("__"+senderStubName+serverName));
    } catch(Exception e) { throw new SenderStubNotFoundException("*** Troubles in detecting remote sender's stubs: "+"__"+senderStubName+serverName); }
  }

}
