package mChaRM.PremChaRM;

import openjava.mop.*;
import openjava.syntax.*;
import openjava.ptree.*;
import openjava.ptree.util.PartialParser;
import mChaRM.multichannel.*;
import java.io.*;
import java.util.*;

 /**
  * This class represents the OpenJava Preprocessor for a class containing the mChaRM mechanism.
  * Its role is to translate the OpenJava class who instantiates it (which is a class written
  * in an extended syntax of java) into three pure-java classes. The whole class contains methods
  * and objects representing the mChaRM objects <bold>Core Channel</bold>, <bold>Sender</bold> and
  * <bold>Receiver</bold>. The classes created by PremChaRM represent these objects and they are
  * obtained sharing out methods and fields according the use they did. A method or a field is
  * put into a class only if it is really used by respective mChaRM object.
  * <BR>
  * @author Valentina Cord&igrave; (<A HREF="mailto:1996s081@educ.disi.unige.it">1996s081@educ.disi.unige.it</A>)
  * @author Maurizio Mazza (<A HREF="mailto:1996s065@educ.disi.unige.it">1996s065@educ.disi.unige.it</A>)
  * @author Roberto Montagna (<A HREF="mailto:1996s063@educ.disi.unige.it">1996s063@educ.disi.unige.it</A>)
  **/

public class PremChaRM extends OJClass  
 {
/*---------------------------------------
 Public Constants
---------------------------------------*/

/*---------------------------------------
 Private Constants
---------------------------------------*/
 /**
  * This constant represent the kind.
 **/
 private final static int KIND = 0;

 /**
  * This constant represent the core.
 **/
 private final static int CORE = 1;

  /**
  * This constant represent the sender.
  **/
 private final static int SENDER = 2;

  /**
  * This constant represent the receiver.
  **/
 private final static int RECEIVER = 3;

  /**
  * This constant represent name class.
  * it is an index of memory structure that contains
  * information about index classes
  **/
 private final static int CLASSNAME = 0;

  /**
  * This constant represent the enriches clause.
  * It used during the parsing of added syntax.
  **/
 private final static int ENRICHNAME = 1;

  /**
  * This constant represent the provides clause.
  * It used during the parsing of added syntax.
  **/
 private final static int PROVIDESNAME = 2;

 /**
  * This array of constant contains the names of the base methods.
  * This array is indexed over the constants:
  * <UL> 
  * <LI>{@link #CORE <B>CORE</B>}</LI>
  * <LI>{@link #SENDER <B>SENDER</B>}</LI>
  * <LI>{@link #RECEIVER <B>RECEIVER</B>}</LI>
  * </UL>
  **/
 private final static String[] NAME_START_METHODS = {"","coreMetaBehavior","senderSideMetaBehavior","receiverSideMetaBehavior"};

 /**
  * This array of constant contains the names of base classes.
  * This class is used to default enrich.
  * This array is indexed over the constants:
  * <UL> 
  * <LI>{@link #CORE <B>CORE</B>}</LI>
  * <LI>{@link #SENDER <B>SENDER</B>}</LI>
  * <LI>{@link #RECEIVER <B>RECEIVER</B>}</LI>
  * </UL>
  **/
 private final static String[] NAME_BASE_CLASSES = {"","mChaRM.multichannel.channelCore","mChaRM.multichannel.senderStub","mChaRM.multichannel.receiverStub"};

/*---------------------------------------
 Private Fields
---------------------------------------*/
 /**
  * This OJSubClass contains, if present, the new class CORE
  **/
 private OJSubClass ojsCore;

 /**
  * This OJSubClass contains, if present, the new class SENDER
  **/
 private OJSubClass ojsSender;

 /**
  * This OJSubClass contains, if present, the new class RECEIVER
  **/
 private OJSubClass ojsReceiver;

 /**
  * This Vector contains all the method declared in base class
  **/
 private Vector ojMethods = new Vector();

 /**
  * This Vector contains the method of hierarchy of the Core.
  * It contains all methods of all class, beginnig to enriches CORE
  * ending with {@link #NAME_BASE_CLASSES <B>NAME_BASE_CLASSES</B>} relative
  **/
 private Vector ojMethodsCore = new Vector();

 /**
  * This Vector contains the method of hierarchy of the Sender.
  * It contains all methods of all class, beginnig to enriches SENDER
  * ending with {@link #NAME_BASE_CLASSES <B>NAME_BASE_CLASSES</B>} relative
  **/
 private Vector ojMethodsSender   = new Vector();

 /**
  * This Vector contains the method of hierarchy of the Receiver.
  * It contains all methods of all class, beginnig to enriches RECEIVER
  * ending with {@link #NAME_BASE_CLASSES <B>NAME_BASE_CLASSES</B>} relative
  **/
 private Vector ojMethodsReceiver = new Vector();

 /**
  * This array contains all methods of create Core class.
  * It use during a translating main class, to check if the methods
  * invoke over an instance of Start class are present in Core class generated.
  **/
 private OJMethod[] coreMethods;

 /**
  * This array contains name of output class create during a translating process.
  * It use during a translating of main class, to find a name of new class and
  * change the allocatition and invocation of static methods inside the main class.
  **/
 private String[] OJMIclassNames = new String[4];

 /**
  * Instance of Graph, is used to check the consistence of an access to fields.
  * The prepocessor insert the information into the graph using some utils
  * and check the consistence of an access to fields using member function of PreGraph.
  **/
 private PreGraph ojGraph = new PreGraph();


 /**
  * This variable contains the parse tree reapresentanting the extedend 
  * syntax. It contains all information about name of output class and name of enriches
  * class.
  **/
 private ObjectList ExtendedSyntax;


 /**
  * This variable contains all fields of starting class.
  **/
 private OJField[] ojFields;

 /**
  * Every cell of array meaning that relative output Class is defined.
  **/
 private boolean[] isDefined = {false, false, false};


 /**
  * During a process that translate class that contains main, is possible
  * that in several situation the programm need information contained in
  * OJMI, this flag used in combination of function {@link #HandleFirstTime( String ) <B>HandleFirstTime( String )</B>}
  * permitt to access to information only one time.
  **/
 private boolean firstTime = true;
/*---------------------------------------
 Extended Syntax Parser
---------------------------------------*/

 /**
  * Function predefined in OJClass, is used to check if the
  * keyword present in added syntax regard current OJClass
  * <HR>
  * @param keyword String that must be check
  * @return true if the keyword is correct
  **/
 public static boolean isRegisteredKeyword(String keyword) 
  {
  PreUtils.printDebug("Start Method isRegisteredKeyword", PreUtils.dbgSyntax, 0);

  if (keyword.equals("kind")) return true;
    return OJClass.isRegisteredKeyword( keyword );
  }


 /**
  * Function predefined in OJClass, is used to get a parse tree of
  * added syntax. 
  * <HR>
  * @param keyword start String of added syntax
  * @return the parse tree contains all added syntax
  **/
 public static SyntaxRule getDeclSuffixRule(String keyword) 
  {
  PreUtils.printDebug("Start Method getDeclSuffixRule", PreUtils.dbgSyntax, 0);

  if (keyword.equals("kind"))
    {
    SyntaxRule syntax[] = new SyntaxRule[4];
    syntax[0] = new NameRule();
    syntax[1] = new IterationRule(
                     new SelectionRule( 
	                 new CompositeRule( 
		             new PrepPhraseRule( "core", new NameRule() ),
		             new PrepPhraseRule( "enriches", new TypeNameRule() ),
		             new PrepPhraseRule( "provides", new DefaultListRule( new TypeNameRule(), TokenID.COMMA) ) ),
		         new PrepPhraseRule( "core", new NameRule() ) ), true );
    syntax[2] = new IterationRule(
                     new SelectionRule( 
	                 new CompositeRule( 
		             new PrepPhraseRule( "senders", new NameRule() ),
		             new PrepPhraseRule( "enriches", new TypeNameRule() ),
		             new PrepPhraseRule( "provides", new DefaultListRule( new TypeNameRule(), TokenID.COMMA) ) ),
		         new PrepPhraseRule( "senders", new NameRule() ) ), true );
    syntax[3] = new IterationRule(
                     new SelectionRule( 
	                 new CompositeRule( 
		             new PrepPhraseRule( "receivers", new NameRule() ),
		             new PrepPhraseRule( "enriches", new TypeNameRule() ),
		             new PrepPhraseRule( "provides", new DefaultListRule( new TypeNameRule(), TokenID.COMMA) ) ),
		         new PrepPhraseRule( "receivers", new NameRule() ) ), true );
    return new CompositeRule( syntax );
    }
  return OJClass.getDeclSuffixRule( keyword );
  }

/*-------------------------------------
 Open Output class
-------------------------------------*/

 /**
  * This method create the output class. It starting after that the
  * added syntax is parsing, it create and initialize the output class.
  * <HR>
  * The new class is insert ipn instance of {@link OJSubClass <B>OJSubClass</B>}<BR>
  * @see OJSubClass OJSubClass 
  **/
 public void openClasses()
  {
  PreUtils.printDebug("Start Method openClasses", PreUtils.dbgOpenClasses, 0);
  String strClassName, strSuperClassName;

  PreUtils.setName( getName() );

  try{
    OJClass ojcSuperClass;
    ObjectList olSyntax, olInterfaces;
    String strImports[], strImplements[]=null;

    ParseTreeObject compUnit = getSourceCode(); 
    do{
      compUnit = compUnit.getParent();
      }while( !( compUnit instanceof CompilationUnit ) );
    strImports = ((CompilationUnit)compUnit).getDeclaredImports();

    putMetaInfo( "Kind", ExtendedSyntax.get(KIND).toString() ); 
/* Core Handling */
    PreUtils.printDebug("Core Handling", PreUtils.dbgOpenClasses, 1);

    olSyntax = (ObjectList)ExtendedSyntax.get(CORE);
    if( olSyntax.size() == 0 )
      {
      ojsCore = null;
      putMetaInfo( "Core", NAME_BASE_CLASSES[CORE] );
      }
    else if( olSyntax.size() == 1 )
      {
      if( olSyntax.get(0) instanceof ObjectList )
	{
	strClassName = ((ObjectList)olSyntax.get(0)).get(CLASSNAME).toString();
	strSuperClassName = ((ObjectList)olSyntax.get(0)).get(ENRICHNAME).toString();
        olInterfaces = (ObjectList)((ObjectList)olSyntax.get(0)).get(PROVIDESNAME);
	strImplements = new String[olInterfaces.size()+1];
	for( int i=0; i<strImplements.length-1; i++ )
	  strImplements[i] = olInterfaces.get(i).toString();
	strImplements[olInterfaces.size()] = "mChaRM.multichannel.channelInterface";
	}
       else
	{
	strClassName = olSyntax.get(0).toString();
	strSuperClassName = NAME_BASE_CLASSES[CORE];
	}

      ojsCore = new OJSubClass(strClassName, strSuperClassName, strImplements, strImports );
      putMetaInfo( "Core", strClassName ); 
      }
    else
      {
      System.err.println( "FATAL ERROR: too many 'core' clauses " );
      System.exit( -1 );
      }

/* Sender Handling */
    PreUtils.printDebug("Sender Handing", PreUtils.dbgOpenClasses, 1);

    olSyntax = (ObjectList)ExtendedSyntax.get(SENDER);
    strImplements = null;
    if( olSyntax.size() == 0 )
      {
      ojsSender = null;
      putMetaInfo( "Sender", NAME_BASE_CLASSES[SENDER] );
      }
    else if( olSyntax.size() == 1 )
      {
      if( olSyntax.get(0) instanceof ObjectList )
	{
	strClassName = ((ObjectList)olSyntax.get(0)).get(CLASSNAME).toString();
	strSuperClassName = ((ObjectList)olSyntax.get(0)).get(ENRICHNAME).toString();
	olInterfaces = (ObjectList)((ObjectList)olSyntax.get(0)).get(PROVIDESNAME);
	strImplements = new String[olInterfaces.size()];
	for( int i=0; i<strImplements.length; i++ )
	  strImplements[i] = olInterfaces.get(i).toString();
        }
       else
	{
	strClassName = olSyntax.get(0).toString();
	strSuperClassName = NAME_BASE_CLASSES[SENDER];
	}
      ojsSender = new OJSubClass(strClassName, strSuperClassName, strImplements, strImports );
      putMetaInfo( "Sender", strClassName ); 
      }
    else
      {
      System.err.println( "FATAL ERROR: too many 'sender' clauses " );
      System.exit( -1 );
      }

/* Receiver Handling */
    PreUtils.printDebug("Receiver Handing", PreUtils.dbgOpenClasses, 1);

    olSyntax = (ObjectList)ExtendedSyntax.get(RECEIVER);
    strImplements = null;
    if( olSyntax.size() == 0 )
      {
      ojsReceiver = null;
      putMetaInfo( "Receiver", NAME_BASE_CLASSES[RECEIVER] );
      }
    else if( olSyntax.size() == 1 )
      {
      if( olSyntax.get(0) instanceof ObjectList )
	{
	strClassName = ((ObjectList)olSyntax.get(0)).get(CLASSNAME).toString();
	strSuperClassName = ((ObjectList)olSyntax.get(0)).get(ENRICHNAME).toString();
	 olInterfaces = (ObjectList)((ObjectList)olSyntax.get(0)).get(PROVIDESNAME);
	strImplements = new String[olInterfaces.size()];
	for( int i=0; i<strImplements.length; i++ )
	  strImplements[i] = olInterfaces.get(i).toString();
        }
       else
	{
	strClassName = olSyntax.get(0).toString();
	strSuperClassName = NAME_BASE_CLASSES[RECEIVER];
	}
      ojsReceiver = new OJSubClass(strClassName, strSuperClassName, strImplements, strImports );
      putMetaInfo( "Receiver", strClassName ); 
      }
    else
      {
      System.err.println( "FATAL ERROR: too many 'receiver' clauses " );
      System.exit( -1 );
      }
    }
  catch( Exception e )
    {
    PreUtils.printExceptionText( "FATAL ERROR: Unable to create File ", e );
    System.exit( -1 );
    }
  }

/*-------------------------------------
 Prepare classes
-------------------------------------*/

  /**
   * This method makes two different works: first, it initializes the three
   * output classes inserting import clauses and the relative constructor;
   * then it creates graphs for methods and fields, finally inserts right fields
   * and methods in relative class.
   * In the first phase it finds all method belong to the superclass o the three output class
   * calling {@link #cascadeMethods(OJClass) <B>cascadeMethods</B>}, finally it create the three constructor.
   * In the last phase it divides all member of start class, to do that first find late Binding methods and then 
   * for each method founded search all method called in. At this time for each method founded search all fields 
   * used in. If there are multi-referred fields it indicate a Fatal  Error and terminate the esecution otherwise 
   * it splits the whole classe in three classes.
  **/
private void prepareClasses()
  {
  PreUtils.printDebug("Start Method prepareClasses", PreUtils.dbgPrepClasses, 0);
  PreUtils.printDebug("invoke getDeclaredMethods", PreUtils.dbgPrepClasses, 1);
  ojMethods = PreUtils.array2Vector( getDeclaredMethods() );
  ParameterList plParams; 

  PreUtils.printDebug("Find all superclass method", PreUtils.dbgPrepClasses, 1);
  if( ojsCore != null )
    {
    ojMethodsCore = cascadeMethods( ojsCore.getSuperClass() );
    isDefined[CORE-1] = true;
    }
    
  if( ojsSender != null )
    {
    ojMethodsSender = cascadeMethods( ojsSender.getSuperClass() );
    isDefined[SENDER-1] = true;
    }

  if( ojsReceiver != null )
    {
    ojMethodsReceiver = cascadeMethods( ojsReceiver.getSuperClass() );
    isDefined[RECEIVER-1] = true;
    }
  
  PreUtils.printDebug("invoke getDeclaredFields", PreUtils.dbgPrepClasses, 1);
  ojFields = getDeclaredFields();

  PreUtils.printDebug("Handling Constructors", PreUtils.dbgPrepClasses, 1);
  for( int i=1; i<=RECEIVER; i++ )
    if( isDefined[i-1] )
      insertConstructorMethods(i);
  
  PreUtils.printDebug("Late Binding Methods", PreUtils.dbgPrepClasses, 1);
  for( int i=1; i<=RECEIVER; i++ )
    if( isDefined[i-1] )
      insertLateBindingMethods(i);
  
  for( int i=0; i<ojGraph.numberOfMethods(); i++ )
    PreUtils.printDebug("---------- methods "+ojGraph.methodAt(i)+
                      ojGraph.methodUsedBy(i, CORE ) + " " +
                      ojGraph.methodUsedBy(i, SENDER ) + " " +
                      ojGraph.methodUsedBy(i, RECEIVER ), PreUtils.dbgPrepClasses, 1);

  PreUtils.printDebug("For every constructor search used field", PreUtils.dbgPrepClasses, 1);
  Vector constr=null;
  int[] className=null;
  for( int j=CORE; j<=RECEIVER; j++ )
    {
    switch( j )
      {
      case CORE:
        if( !isDefined[j-1] )
          continue;
        constr = ojsCore.constructors;
	className = new int[] {1,0,0};
        break;
      case SENDER:
        if( !isDefined[j-1] )
          continue;
        constr = ojsSender.constructors;
	className = new int[] {0,1,0};
        break;
      case RECEIVER:
        if( !isDefined[j-1] )
          continue;
        constr = ojsReceiver.constructors;
	className = new int[] {0,0,1};
        break;
      }
    for( int i=0; i<constr.size(); i++ )
      {
      OJMethod tempMethod = (OJMethod)constr.elementAt(i);
      StatementList tempBody;
      String[] parameters;
      try{
        tempBody = tempMethod.getBody();
        parameters = tempMethod.getParameters();
        PreUtils.printDebug("Checked Constructor "+tempMethod, PreUtils.dbgPrepClasses, 1);
        Vector multiRefered = PreUtils.fieldHandling( tempBody, PreUtils.array2Vector(parameters), PreUtils.array2Vector( ojFields ) );
        PreUtils.printDebug("End checked Constructor "+multiRefered, PreUtils.dbgPrepClasses, 1);
        for( int z=0; z<multiRefered.size(); z++ )
          ojGraph.insertFields((OJField)multiRefered.elementAt(z), className );

        for( int z=0; z<ojGraph.numberOfFields(); z++ )
          PreUtils.printDebug("Dangerous Field " + ojGraph.fieldAt(z) + 
	                      " used by " + 
			      ojGraph.fieldUsedBy(z, CORE ) + " " + 
			      ojGraph.fieldUsedBy(z, SENDER ) + " " +
			      ojGraph.fieldUsedBy(z, RECEIVER ), PreUtils.dbgPrepClasses, 2);

        }
      catch( CannotAlterException e )
        {
        PreUtils.printExceptionText( "FATAL ERROR: Unexpected Error, in prepareClasses", e );
        System.exit(-1);
        }
      }
    }

  PreUtils.printDebug("For every methods search used field", PreUtils.dbgPrepClasses, 1);
  for( int i=0; i<ojGraph.numberOfMethods(); i++ )
    {
    OJMethod tempMethod = (ojGraph.methodAt(i));
    StatementList tempBody;
    String[] parameters;
    try{
      tempBody = tempMethod.getBody();
      parameters = tempMethod.getParameters();
      PreUtils.printDebug("Checked Method "+tempMethod, PreUtils.dbgPrepClasses, 1);
      Vector multiRefered = PreUtils.fieldHandling( tempBody, PreUtils.array2Vector(parameters), PreUtils.array2Vector( ojFields ) );
      PreUtils.printDebug("End checked Method "+multiRefered, PreUtils.dbgPrepClasses, 1);
      for( int j=0; j<multiRefered.size(); j++ )
        ojGraph.insertFields((OJField)multiRefered.elementAt(j), ojGraph.methodUsedBy(i) );

      for( int j=0; j<ojGraph.numberOfFields(); j++ )
        PreUtils.printDebug("Dangerous Field " + ojGraph.fieldAt(j) + 
	                    " used by " + 
			    ojGraph.fieldUsedBy(j, CORE ) + " " + 
			    ojGraph.fieldUsedBy(j, SENDER ) + " " +
			    ojGraph.fieldUsedBy(j, RECEIVER ), PreUtils.dbgPrepClasses, 2);
      }
    catch( CannotAlterException e )
      {
      PreUtils.printExceptionText( "FATAL ERROR: Unexpected Error, in prepareClasses", e );
      System.exit(-1);
      }
    }
   
  Vector ErrorField = ojGraph.DangerousFields();
  if( ErrorField.size() > 0 )
    {
    PreUtils.printDebug("Double referencing Fields "+ErrorField, PreUtils.dbgPrepClasses, 1);
    System.err.println("FATAL ERROR. Double referencing Fields not Permitted");
    System.exit(-1);
    }

  int[] usedIn;
  PreUtils.printDebug("Correct usage of Fields", PreUtils.dbgPrepClasses, 1);
  PreUtils.printDebug("Insert Fields in relative class", PreUtils.dbgPrepClasses, 1);
  for( int i=0; i<ojGraph.numberOfFields(); i++ )
    {
    if( ojGraph.fieldUsedBy( i, CORE ) )
      ojsCore.addNewFields(ojGraph.fieldAt(i));
    if( ojGraph.fieldUsedBy( i, SENDER ) )
      ojsSender.addNewFields(ojGraph.fieldAt(i));
    if( ojGraph.fieldUsedBy( i, RECEIVER ) )
      ojsReceiver.addNewFields(ojGraph.fieldAt(i));
    }
  PreUtils.printDebug("Insert methods in relative class", PreUtils.dbgPrepClasses, 1);
  for( int i=0; i<ojGraph.numberOfMethods(); i++ )
    {
    if( ojGraph.methodUsedBy( i, CORE ) )
      ojsCore.addNewMethods(ojGraph.methodAt(i));
    if( ojGraph.methodUsedBy( i, SENDER ) )
      ojsSender.addNewMethods(ojGraph.methodAt(i));
    if( ojGraph.methodUsedBy( i, RECEIVER ) )
      ojsReceiver.addNewMethods(ojGraph.methodAt(i));
    }
  }

/*-------------------------------------
 cascadeMethods
-------------------------------------*/

 /**
  * This method return all method declared in a class hierarchy.
  * To do that it call itself recursively, stops when find a Base class.
  * <HR>
  * @param ojClass Class whom find methods
  * @return Vector contains all methods declared
  * @see #NAME_BASE_CLASSES array that contains base class
 **/
 private Vector cascadeMethods( OJClass ojClass ) 
  {
  PreUtils.printDebug("Start Method cascadeMethods " + ojClass, PreUtils.dbgPrepClasses, 0);
  Vector methods = PreUtils.array2Vector( (Object[])ojClass.getDeclaredMethods() );

  if( !ojClass.getName().equals( NAME_BASE_CLASSES[CORE] ) &&
      !ojClass.getName().equals( NAME_BASE_CLASSES[SENDER])  &&
      !ojClass.getName().equals( NAME_BASE_CLASSES[RECEIVER] ) )

     methods.addAll( cascadeMethods( ojClass.getSuperclass() ) );

  return methods;
  }

/*-------------------------------------
 insertConstructorMethods
-------------------------------------*/

 /**
  * This method finds particulary methods representing constructors.
  * @param className name of class where it search the overridden methods
 **/
 private void insertConstructorMethods(int className)
  {
  PreUtils.printDebug("Start Method insertConstructorMethods " + className, PreUtils.dbgPrepClasses, 0);
  OJSubClass myClass = null;
  String[] ConstructorParameters = new String[0], ConstructorParameters1 = new String[0];
  OJMethod method=null;
  OJSubClass ojClass=null;
  int[] vUsed=null;
  boolean baseCostructorFound = false;

  Vector myMethods=null;
  switch( className )
    {
    case CORE: 
      myClass = ojsCore;
      ConstructorParameters = new String[] {"java.lang.String","java.lang.String[]","java.lang.String","java.lang.String"};
      vUsed = new int[] {1,0,0};
      ojClass = ojsCore;
      break;
    case SENDER:
      myClass = ojsSender;
      ConstructorParameters = new String[] {"java.lang.Object","java.lang.String","java.lang.String"};
      vUsed = new int[] {0,1,0};
      ojClass = ojsSender;
      break;
    case RECEIVER:
      myClass = ojsReceiver;
      ConstructorParameters = new String[] {"java.lang.Object","java.lang.String","java.lang.String"};
      ConstructorParameters1 = new String[] {"uka.karmi.rmi.server.UnicastRemoteObject","java.lang.String","java.lang.String"};
      vUsed = new int[] {0,0,1};
      ojClass = ojsReceiver;
      break;
    }
  for( int i=0; i<ojMethods.size(); i++ )
    {
    method = (OJMethod)ojMethods.elementAt(i);
    if( method.getName().equals( myClass.getName() ) )
      {
      OJClass[] cTipi = method.getParameterTypes();
      String[] strTipi = new String[cTipi.length];
      for( int j=0; j<strTipi.length; j++ )
        strTipi[j] = cTipi[j].getName();
      if( !baseCostructorFound && 
          ( PreUtils.equalsParameterType( ConstructorParameters, strTipi ) || PreUtils.equalsParameterType( ConstructorParameters1, strTipi ) ) )
        baseCostructorFound = true;
      ojClass.addNewConstructor(method);
      }
    }

  if( !baseCostructorFound )
    {
    System.err.println("FATAL ERROR: Default constructor not found");
    System.exit(-1);
    }
  
  try{
    Vector constr = myClass.constructors;
    Vector fstLine = myClass.fstLineConstr;
    for( int i=0; i<constr.size(); i++ )
      {
      insertMethodsFrom(method, className );
      method = (OJMethod)constr.elementAt(i);
      String fLine = (String)fstLine.elementAt(i);
      if( !fLine.equals( "" ) )
        {
        String[] exprList = parseConstr( fLine );

        StatementList stmListTest = new StatementList();
        for( int j=0; j<exprList.length; j++)
          {
          Expression expr = PartialParser.makeExpression(method.getEnvironment(), exprList[j]);
          stmListTest.add( new VariableDeclaration( new TypeName( expr.getType( method.getEnvironment() ).getName() ), "__constr_variable_name"+j, expr ) );
          }
        OJMethod methodTest = new OJMethod(method.getDeclaringClass(), method.getModifiers(), method.getReturnType(), 
                                           "__constr_method", method.getParameterTypes(), method.getParameters(), 
                                           method.getExceptionTypes(), stmListTest);
        insertMethodsFrom( methodTest, className );
        try{
          PreUtils.printDebug("Checked constructor methodTest "+methodTest, PreUtils.dbgPrepClasses, 1);
          Vector multiRefered = PreUtils.fieldHandling( stmListTest, PreUtils.array2Vector(methodTest.getParameters()), PreUtils.array2Vector( ojFields ) );
          PreUtils.printDebug("End checked constructor methodTest "+multiRefered, PreUtils.dbgPrepClasses, 1);
          for( int j=0; j<multiRefered.size(); j++ )
            ojGraph.insertFields((OJField)multiRefered.elementAt(j), vUsed );

          for( int j=0; j<ojGraph.numberOfFields(); j++ )
            PreUtils.printDebug("Dangerous Field " + ojGraph.fieldAt(j) + 
	                        " used by " + 
			        ojGraph.fieldUsedBy(j, CORE ) + " " + 
			        ojGraph.fieldUsedBy(j, SENDER ) + " " +
			        ojGraph.fieldUsedBy(j, RECEIVER ), PreUtils.dbgPrepClasses, 2);
          }
        catch( CannotAlterException e )
          {
          PreUtils.printExceptionText( "FATAL ERROR: Unexpected Error, in prepare constructor", e );
          System.exit(-1);
          }
        }
      }
    }
  catch( CannotAlterException e )
    {
    PreUtils.printExceptionText("FATAL ERROR: Unexpected error in constructor handling",e);
    System.exit(-1);
    }
  catch( MOPException e )
    {
    PreUtils.printExceptionText("FATAL ERROR: Constructor incorrect ",e);
    System.exit(-1);
    }
  catch( Exception e )
    {
    PreUtils.printExceptionText("FATAL ERROR: Unespected Error Constructor incorrect ",e);
    System.exit(-1);
    }

  }


 public String[] parseConstr( String constr )
  {
  PreUtils.printDebug("Start method parse constr String" + constr, PreUtils.dbgPrepClasses, 0);
  PreUtils.printDebug("String to parsing" + constr, PreUtils.dbgPrepClasses, 1);
  StringTokenizer strToken = new StringTokenizer( constr, new String( new char[] { ',', '"', '(', ')', '{', '}' } ), true );
  String tmp = null;
  Vector parameters = new Vector();
  String[] param = null;
  try{
    tmp = strToken.nextToken();
    if( !tmp.equals("this") && !tmp.equals("super") )
      {
      System.err.println("FATAL ERROR: format of constructor incorrect");
      System.exit( -1 );
      }
    tmp = strToken.nextToken();
    if( !tmp.equals("(") )
      {
      System.err.println("FATAL ERROR: format of constructor incorrect");
      System.exit( -1 );
      }
    String ret = new String();
    while( true )
      {
      tmp = strToken.nextToken();
      if( tmp.charAt(0) == ')' ) 
        {
        if( parameters.size() != 0 && ret.length() == 0 )
          {
          System.err.println("FATAL ERROR: Format of constructor incorrect");
          System.exit(-1);
          }
        parameters.add(ret);
        PreUtils.printDebug("il parametro  : " + ret, PreUtils.dbgPrepClasses, 1);
        ret = new String(); 
        break;
        }
      if( tmp.charAt(0) == ',' )
        {
        parameters.add(ret);
        PreUtils.printDebug("il parametro  : " + ret, PreUtils.dbgPrepClasses, 1);
        ret = new String(); 
        continue;
        }
      ret += tmp;
      if( tmp.charAt(0) == '(' )
        {
        ret += parseConstr( strToken, ')' ); 
        continue;
        }
      if( tmp.charAt(0) == '"' )
        {
        ret += parseConstr( strToken, '"' ); 
        continue;
        }
      if( tmp.charAt(0) == '{' ) 
        {
        ret += parseConstr( strToken, '}' ); 
        continue;
        }
      }
    }
  catch( NoSuchElementException e )
    {
    System.err.println("FATAL ERROR: Format of constructor incorrect");
    System.exit(-1);
    }

  param = new String[parameters.size()];
  for( int i=0; i<param.length; i++ )
    param[i] = (String)parameters.elementAt(i);

  for( int i=0; i<param.length; i++ )
    PreUtils.printDebug("il parametri sono : " + param[i], PreUtils.dbgPrepClasses, 1);

  return param;
  }

 public String parseConstr( StringTokenizer strToken, char stop )
  {
  String tmp,ret= new String();
  while( true )
    {
    tmp = strToken.nextToken();
    ret += tmp;
    if( tmp.charAt(0) == stop )
      {
      if( stop == '"' && ret.length() > 1 && ret.charAt(ret.length()-1) == '\\' )
        continue;
      return ret;
      }
    if( tmp.charAt(0) == '(' )
      {
      ret += parseConstr( strToken, ')' ); 
      continue;
      }
    if( tmp.charAt(0) == '"' )
      {
      ret += parseConstr( strToken, '"' ); 
      continue;
      }
    if( tmp.charAt(0) == '{' ) 
      {
      ret += parseConstr( strToken, '}' ); 
      continue;
      }
    }
  }

/*-------------------------------------
 insertLateBindingMethods
-------------------------------------*/

 /**
  * This method find methods, in start class, that override someone else in hierarchy class specified by className.
  * In particulary for each method, in hierarchy class specified by className, it searchs once in start class that is
  * equals to him ( same parameters, same name ).
  * @param className name of class where it search the overridden methods
 **/
 private void insertLateBindingMethods(int className)
  {
  PreUtils.printDebug("Start Method insertLateBindingMethods " + className, PreUtils.dbgPrepClasses, 0);
  Vector myMethods=null;
  String[] interfaces=null;
  switch( className )
    {
    case CORE: 
      myMethods=ojMethodsCore;
      interfaces = ojsCore.getInterfacesStr();
      break;
    case SENDER:
      myMethods=ojMethodsSender;
      interfaces = ojsSender.getInterfacesStr();
      break;
    case RECEIVER:
      myMethods=ojMethodsReceiver;
      interfaces = ojsReceiver.getInterfacesStr();
      break;
    }
  for( int i=0; i<myMethods.size(); i++ )
    for( int j=0; j<ojMethods.size(); j++ )
      if( PreUtils.overriddenMethod( ((OJMethod)myMethods.elementAt(i)), ((OJMethod)ojMethods.elementAt(j)) ) )
        if( ojGraph.insertMethod( (OJMethod)ojMethods.elementAt(j), className ) )
	  insertMethodsFrom((OJMethod)ojMethods.elementAt(j), className);

  try{
    for( int i=0; interfaces!=null && i<interfaces.length; i++ )
      {
      OJMethod[] interMethod = OJClass.forName( interfaces[i] ).getAllMethods();
      for( int j=0; j<interMethod.length; j++ )
        for( int z=0; z<ojMethods.size(); z++ )
          if( PreUtils.overriddenMethod( ((OJMethod)ojMethods.elementAt(z)), interMethod[j] ) )
            if( ojGraph.insertMethod( (OJMethod)ojMethods.elementAt(z), className ) )
	      insertMethodsFrom((OJMethod)ojMethods.elementAt(z), className);
      }
    }
  catch( OJClassNotFoundException e )
    {
    PreUtils.printExceptionText("FATAL ERROR: implemented interface not Found",e);
    System.exit(-1);
    }
  catch( NullPointerException e )
    {
    PreUtils.printExceptionText("FATAL ERROR: unexpected error",e);
    System.exit(-1);
    }
  }
 
 /**
  * This method insert in graph the starting method and all method called in.
  * It does the transitive closure of ojMethod.
  * @param ojMethod starting method to find all methods invoked in.
  * @param className name of class that contains starting method
 **/
 private void insertMethodsFrom( OJMethod ojMethod, int className )
  {
  PreUtils.printDebug("Start Method insertMethodsFrom method " + ojMethod + " " + className, PreUtils.dbgPrepClasses, 0);
  OJMethod tempOJMethod;
  Stack workMethods = new Stack();
  workMethods.push( ojMethod );
 
  while( !workMethods.isEmpty() )
    {
    OJMethod tempMethod = (OJMethod)workMethods.pop();
    Vector ojmTemp = new Vector();
    Vector allEnvs = new Vector();
    PreUtils.getCalledMethods( tempMethod, ojmTemp, allEnvs );
    
    for( int i=0; i<ojmTemp.size(); i++ )
      if( ( tempOJMethod = methodCall2OJMethod((MethodCall)ojmTemp.elementAt(i), (Environment)allEnvs.elementAt(i)) ) != null )
	if( ojGraph.insertMethod( tempOJMethod, className ) )
	  workMethods.push( tempOJMethod );
    }
  }

/*-------------------------------------
 methodCall2OJMethod
-------------------------------------*/

 /**
  * This method perform the translation of MethodCall into OJMethod.
  * This translation is done only if the MethodCall deals a method belong to start class, otherwise
  * it return null.
  * To do the translation it searchs a OJMethod with same name and same parameters type.
  * @param mcMethod MethodCall to translate in OJMethod
  * @param environment current Environment
  * @return OJMethod invoked by specified MethodCall.
 **/
 public OJMethod methodCall2OJMethod( MethodCall mcMethod, Environment environment )
  {
  PreUtils.printDebug("Start Method methodCall2OJMethod - methodCall: " + mcMethod, PreUtils.dbgPrepClasses, 0);
  Expression referExpr;
  OJMethod returnMethod = null;
  String name = mcMethod.getName();
  int i;
  boolean correctMethod = true;

  if( !( (referExpr = mcMethod.getReferenceExpr()) instanceof SelfAccess && 
         ((SelfAccess)referExpr).getAccessType() == SelfAccess.SUPER) )
    {
    PreUtils.printDebug("interesting method", PreUtils.dbgPrepClasses, 1);
    ExpressionList lexp = mcMethod.getArguments();
    try{
      if( lexp.size() == 0 )
        {
        PreUtils.printDebug("method without parameters", PreUtils.dbgPrepClasses, 1);
        for( i=0; i<ojMethods.size(); i++)
          if( ((OJMethod)ojMethods.elementAt(i)).getParameters().length == 0 && name.equals( (returnMethod = (OJMethod)(ojMethods.elementAt(i))).getName() ) )
            return returnMethod;
        }
       else
        {
        PreUtils.printDebug("method with parameters", PreUtils.dbgPrepClasses, 1);
        OJClass[] types = new OJClass[lexp.size()];
        for( i=0; i<lexp.size(); i++)
          types[i]= lexp.get(i).getType(environment); 
        for( i=0; i<ojMethods.size(); i++)
          {
          PreUtils.printDebug("compare with "+(OJMethod)ojMethods.elementAt(i), PreUtils.dbgPrepClasses, 2);
          OJClass[] temp = ((OJMethod)ojMethods.elementAt(i)).getParameterTypes();
          if( name.equals( (returnMethod = (OJMethod)(ojMethods.elementAt(i))).getName() ) &&  temp.length == types.length )
            {
            for( int j=0; j< temp.length; j++ )
              if( !temp[j].equals(types[j]) )
                {
                PreUtils.printDebug("method different", PreUtils.dbgPrepClasses, 2);
                correctMethod = false;
                break;
                } 
            if(correctMethod)
              {
              PreUtils.printDebug("method equals", PreUtils.dbgPrepClasses, 2);
              return returnMethod;
              }
            correctMethod = true; 
            }
          }   
        }     
      }
    catch( CannotAlterException e )
      {
      PreUtils.printExceptionText("Unexpected FATAL ERROR, in MethodCall2OJMethod", e );
      System.exit(-1);
      }
    catch( Exception e )
      {
      PreUtils.printExceptionText("Unexpected generic FATAL ERROR",e);
      System.exit(-1);
      }
    }
  return null;
  }

/*-------------------------------------
 prepareMainClass
-------------------------------------*/

 /**
  * This method delete main class and add three instance of the generated classes.
  * It does that to force the classes compiling
 **/
 private void prepareMainClass()
  {
  PreUtils.printDebug("Start Method prepareMainClass", PreUtils.dbgPrepClasses, 0);
  try{
    OJMethod[] myMethod = getDeclaredMethods();
    for(int i=0; i<myMethod.length; i++)
      removeMethod( myMethod[i] );

    OJField[] myField = getDeclaredFields();
    for(int i=0; i<myField.length; i++)
      removeField( myField[i] );

    OJConstructor[] myConstructor = getDeclaredConstructors();
    for(int i=0; i<myConstructor.length; i++)
      removeConstructor( myConstructor[i] );

    if( isDefined[CORE-1] ) 
      {
      OJClass newTypeCore = OJClass.forClass(Object.class);
      OJSystem.env.record(ojsCore.getName(), newTypeCore);
      addField( new OJField(getEnvironment(), this, 
                            new FieldDeclaration( new ModifierList(ModifierList.PUBLIC),
                            new TypeName( ojsCore.getName() ),
                            new VariableDeclarator( "Core", null ) ) ) );
      }

    if( isDefined[SENDER-1] )
      {
      OJClass newTypeServer = OJClass.forClass(Object.class);
      OJSystem.env.record(ojsSender.getName(), newTypeServer );
      addField( new OJField(getEnvironment(), this, 
                            new FieldDeclaration( new ModifierList(ModifierList.PUBLIC),
                            new TypeName( ojsSender.getName() ),
                            new VariableDeclarator( "Sender", null ) ) ) );
      }

    if( isDefined[RECEIVER-1] )
      {
      OJClass newTypeReceiver = OJClass.forClass(Object.class);
      OJSystem.env.record(ojsReceiver.getName(), newTypeReceiver );
      addField( new OJField(getEnvironment(), this, 
                            new FieldDeclaration( new ModifierList(ModifierList.PUBLIC),
                            new TypeName( ojsReceiver.getName() ),
                            new VariableDeclarator( "Receiver", null ) ) ) );
      }

    setSuperclass( OJClass.forName("java.lang.Object") );
    }
  catch( OJClassNotFoundException e )
    {
    PreUtils.printExceptionText("FATAL ERROR, Object not found",e);
    System.exit(-1);
    }
  catch( CannotAlterException e )
    {
    PreUtils.printExceptionText("FATAL ERROR, unable to modify own class",e);
    System.exit(-1);
    }
  }

/*-------------------------------------
 printClasses
-------------------------------------*/

 /**
  * This method write on file the code of the three classes if they are defining.
 **/
 private void printClasses()
  {
  PreUtils.printDebug("Start Method printClasses", PreUtils.dbgPrintClasses, 0);
  if( isDefined[CORE-1] )
    ojsCore.printOutputClass();
  if( isDefined[SENDER-1] )
    ojsSender.printOutputClass();
  if( isDefined[RECEIVER-1] )
    ojsReceiver.printOutputClass();
  }

/*---------------------------------------
 translateDefinition
---------------------------------------*/

 /**
  * This is the starting point of pre-compiling.
  * Initially it checks th syntax, then creates and initializes the output classes, finally it prints\
  * all three classes and prepare main class. 
  * All this things are doing using several methods.
 **/
 public void translateDefinition() 
  {
  PreUtils.printDebug("Start Method translateDefinition", PreUtils.dbgSyntax, 0);

  PreUtils.printDebug("invoke getSuffix", PreUtils.dbgSyntax, 1);
  ExtendedSyntax = (ObjectList)this.getSuffix("kind");

  PreUtils.printDebug("invoke openClasses", PreUtils.dbgSyntax, 1);
  openClasses();

  PreUtils.printDebug("invoke prepareClasses", PreUtils.dbgSyntax, 1);
  prepareClasses();

  PreUtils.printDebug("invoke printClasses", PreUtils.dbgSyntax, 1);
  printClasses();

  PreUtils.printDebug("invoke prepareMainClasses", PreUtils.dbgSyntax, 1);
  prepareMainClass();
  }

/*---------------------------------------
 expandVariableDeclaration
---------------------------------------*/

 /**
  * This method is invoked on every VariableDeclaration instance of start class. 
  * It change type of Declaration into Core Type.
  * @param env current Environment
  * @param var current VariableDeclaration
 **/
 public Statement expandVariableDeclaration( Environment env, VariableDeclaration var )
  {
  PreUtils.printDebug("Start Method expandVardecl", PreUtils.dbgMethodCall, 0);
  HandleFirstTime( var.getTypeSpecifier().getName() );
  var.setTypeSpecifier( new TypeName( OJMIclassNames[CORE] ) );
  return var;
  }

/*---------------------------------------
 expandTypeName
---------------------------------------*/

 /**
  * This method is invoked on every TypeName instance of start class. 
  * It change type name into Core Type.
  * @param env current Environment
  * @param type current TypeName
 **/
 public TypeName expandTypeName(Environment env, TypeName type) 
  {
  PreUtils.printDebug("Start Method expandTypeName", PreUtils.dbgMethodCall, 0);
  HandleFirstTime( type.getName() );

  if( type.getParent() instanceof AllocationExpression ||
      type.getParent() instanceof VariableDeclaration )
    return type;

  if( type.getName().equals( getName() ) )
    type.setName( OJMIclassNames[CORE] );
  
  return type;
  }

/*---------------------------------------
 expandAllocation
---------------------------------------*/

 /**
  * This method is invoked on every AllocationExpression instance of start class.
  * It earch the constructor and change it into a new constructor of Core class.
  * To do that change name of current constructor and add it three parameters, KIND of Channel
  * name of SENDER and RECEIVER.
  * @param env current Environment
  * @param expr current AllocationExpression
 **/
 public Expression expandAllocation( Environment env, AllocationExpression expr )
  {
  PreUtils.printDebug("Start Method expandAllocation", PreUtils.dbgAllocation, 0);

  HandleFirstTime( expr.getClassType().getName() );
  if( expr.getClassType().getName().equals( getName() ) )
    {
    expr.setClassType( new TypeName( OJMIclassNames[CORE] ) );
    ExpressionList paramList = new ExpressionList();
    paramList.add( new Literal( Literal.STRING, "\""+OJMIclassNames[KIND]+"\"" ) );
    paramList.addAll( expr.getArguments() );
    paramList.add( new Literal( Literal.STRING, "\""+OJMIclassNames[SENDER]+"\"" ) );
    paramList.add( new Literal( Literal.STRING, "\""+OJMIclassNames[RECEIVER]+"\"" ) );
    expr.setArguments( paramList );
    }
  PreUtils.printDebug("New expression made: "+expr, PreUtils.dbgAllocation, 2);
  return expr;
  }

/*---------------------------------------
 expandMethodCall
---------------------------------------*/

 /**
  * This method is invoked on every MethodCall instance of start class   not tested yet. 
  * It check if a method is static, if so it translate it, otherwise don't make anything.
  * To translate calling {@link #changeReferenceExpr(Expression,Environment) <B>changeReferenceExpr</B>}
  * @param env current Environment
  * @param call current MethodCall
 **/
 public Expression expandMethodCall( Environment env, MethodCall call )
  {
  PreUtils.printDebug("Start Method expandMethodCall", PreUtils.dbgMethodCall, 0);

  if( call.getReferenceType() == null )
    return call;

  HandleFirstTime( call.getReferenceType().getName() );
  boolean find = false;
  OJMethod method = methodCall2OJMethod( call, env );
  if( coreMethods != null )
    {
    for( int i=0; i<coreMethods.length ; i++ )
      if( coreMethods[i].equals( method ) ) 
	{
	if( coreMethods[i].getModifiers().isStatic() )
	  changeReferenceExpr( call, env );
	find = true;  
	break;
	}  
    if( find == false )	
      {
      System.err.println("FATAL ERROR: Wrong Method Access ");
      System.exit(-1);
      }
    } 
  return call;  	  
  }

/*---------------------------------------
 HandleFirstTime
---------------------------------------*/

 /**
  * This method takes the info in OJMI file of sterting class.
  * This method is performs only once.
  * @param expr string representig the Core name
 **/
 public void HandleFirstTime( String expr )
  {
  PreUtils.printDebug("Start Method HandleFirstTime", PreUtils.dbgAllocation, 0);

  if( firstTime )
    {
    try{
      OJClass CoreClass = OJClass.forName( expr );
      coreMethods = CoreClass.getAllMethods();
      OJMIclassNames[KIND] = CoreClass.getMetaInfo("Kind");
      OJMIclassNames[CORE] = CoreClass.getMetaInfo("Core");
      OJMIclassNames[SENDER] = CoreClass.getMetaInfo("Sender");
      OJMIclassNames[RECEIVER] = CoreClass.getMetaInfo("Receiver");
      firstTime = false;
      }
    catch( OJClassNotFoundException e )
      {
      PreUtils.printExceptionText("FATAL ERROR: Unable to find Core Class ", e);
      System.exit(-1);
      }
    }
  }

 /**
  * This method do a recursive parsing to find typeName to change.
  * When find it delete it, because an instance of class can access 
  * to static member.
  * @param call Expression 
  * @param env current Environment
 **/
private void changeReferenceExpr( Expression call, Environment env )
 {
 if( call instanceof MethodCall )
   {
   Expression RefExpr = ((MethodCall)call).getReferenceExpr();
   TypeName RefType = ((MethodCall)call).getReferenceType();
   if( RefExpr != null )
     changeReferenceExpr( RefExpr, env ); 
    else if( !(call instanceof ArrayAccess) && RefExpr != null && RefType.getName().equals(OJMIclassNames[CORE]) )
     ((MethodCall)call).setReferenceType( new TypeName(getMetaInfo("Core")) ); 
   }  

 if( call instanceof FieldAccess )
   {
   Expression RefExpr = ((FieldAccess)call).getReferenceExpr();
   TypeName RefType = ((FieldAccess)call).getReferenceType();
   if( RefExpr != null )
     changeReferenceExpr( RefExpr, env ); 
    else if( !(call instanceof ArrayAccess) && RefExpr != null && RefType.getName().equals(OJMIclassNames[CORE]) )
     ((FieldAccess)call).setReferenceType( new TypeName(getMetaInfo("Core")) ); 
   }  

 if( call instanceof ArrayAccess )
   {
   Expression RefExpr = ((ArrayAccess)call).getReferenceExpr();
   if( RefExpr != null )
     changeReferenceExpr( RefExpr, env ); 
   }  
 }




 public PremChaRM( openjava.mop.Environment oj_param0, openjava.mop.OJClass oj_param1, openjava.ptree.ClassDeclaration oj_param2 )
  {
  super( oj_param0, oj_param1, oj_param2 );
  }


 public PremChaRM( java.lang.Class oj_param0, openjava.mop.MetaInfo oj_param1 )
  {
  super( oj_param0, oj_param1 );
  }

 }

