package mChaRM.multichannel;

import uka.karmi.rmi.server.*;
import uka.karmi.rmi.*;
import java.net.MalformedURLException;
import java.io.*;

  /**
   * Instances of this class are used as stubs of the multi-channel on the sender side.
   * Each sender which asks a reflective computation to a multi-channel will be extended by a stub of such a multi-channel.<BR> 
   * This class defines the meta-computations that must be performed on the sender side. 
   * Each new meta-behavior to be performed on the sender side can be defined extending this
   * class and overrinding the method {@link #senderSideMetaBehavior <B>senderSideMetaBehavior</B>}.
   * @author  Walter Cazzola (<A HREF="mailto:cazzola@disi.unige.it">cazzola@disi.unige.it</A>)
   * @version 1.2
   * @since Version 1.0
   **/

public class senderStub extends stub implements senderStubInterface {

   /** senderStub constructor.
    * it initialize the stub, registers it as a remote server, and connect it to the core of the multi-channeli, and vice versa.
    * @param myReferent a representant of the stub referent.
    * @param myKind the multi-channel's name.
    * @param myReferentName the referent's name.
    * @exception CoreNotFoundException thrown when the core of the multi-channel, which the stub is part of, 
    *            doesn't exist or isn't correctly registered as a server. 
    * @exception SenderStubCannotBeRegisteredAsAServerException thrown when the stub we are creating cannot be registerd as a server.
    * @exception SenderStubNotFoundException thrown when the core can't link to the this sender stub.
    **/
  public senderStub(Object myReferent, String myKind, String myReferentName) throws SenderStubNotFoundException, CoreNotFoundException, SenderStubCannotBeRegisteredAsAServerException, RemoteException {
    setReferent(myReferent); 
    try {
      Naming.rebind("__"+myReferentName+myKind, this); // register myself 
      setWhoIsMyCore(myKind);
      WhoIsMyCore().senderStubHasBeenCreated(myReferentName);
    } catch(RemoteException e) { 
         throw new SenderStubCannotBeRegisteredAsAServerException("*** Troubles in registering the sender's stub "+myReferentName+" as a RMI server");
    } catch(MalformedURLException e) { 
         throw new SenderStubCannotBeRegisteredAsAServerException("*** Troubles in registering the sender's stub "+myReferentName+" as a RMI server");}
  }
    
   /** This method embodies the reflective behaviour performed on the source locus, and also forwards the trapped request 
    * to the abstract locus.
    * <HR>
    * This method starts on the source locus of the meta-computations related to the reified method call:
    * <CENTER>multiRMI(RsName, methodName, args)</CENTER>
    * then it passes all the stuff to the core of the multi-channel.<BR>
    * the computations performed can make assumption about the fact it will be performed on the sender site.
    * This method can't be neither overrode nor directly called by the programmer.
    * <HR>   
    * Note that receivers can be a subset of the multi-channel's receivers, and as side-effect the {link #senderSideMetaBehavior <B>senderSideMetaBehavior</B>}
    * method can change the arguments passed to the core of channel
    * <HR>
    * @param msg the hijacked method call.
    * @return the result of the method call.
   **/
  final public Object stubBehavior(mChaRMMethodCall msg) {
    try {
      beforeSenderSideMetaBehavior(msg);
      Object res = WhoIsMyCore().coreMetaBehavior(msg);
      msg.setReturnValue(res);
      afterSenderSideMetaBehavior(msg);
    } catch(Exception e) {
      System.out.println("*** Troubles in contacting the channel core");
      e.printStackTrace();
    }
    return msg.getReturnValue();
  }

   /** This method performs part of the sender side meta-behavior.
    * <HR>
    * It is called before passing all the staff to the core and it permits to perform meta-computations on sender side.
    * As default behavior, nothing it is done, it is possible to define new behavior defing a new subclass overriding this method.
    * <HR>
    * @param msg the hijacked method call.
    * @param args the actual arguments of the call.
    **/
  public void beforeSenderSideMetaBehavior(mChaRMMethodCall msg) {}
   
   /** This method performs the sender side meta-behavior.
    * <HR>
    * It is called before passing all the staff to the core and it permits to perform meta-computations on sender side.
    * As default behavior, nothing it is done, it is possible to define new behavior defing a new subclass overriding this method.
    * <HR>
    * @param msg the hijacked method call.
    * @param args the actual arguments of the call.
    **/
  public void afterSenderSideMetaBehavior(mChaRMMethodCall msg) {}

}

