package mChaRM.multichannel;

import uka.karmi.rmi.*;
import java.io.*;
import java.lang.Class;
import java.lang.reflect.*;
import java.net.MalformedURLException;

  /**
   * Instances of this class are used as stubs of the multi-channel on the receiver site.
   * Each receiver connected to a multi-channel will be extended by a stub of such a multi-channel.<BR> 
   * This class defines the meta-computations that have to be performed on the receiver side. 
   * Each new meta-behavior to be performed on the receiver side can be defined extending this class
   * and overrinding the method {@link #receiverSideMetaBehavior <B>receiverSideMetaBehavior</B>}. 
   * @author  Walter Cazzola (<A HREF="mailto:cazzola@disi.unige.it">cazzola@disi.unige.it</A>)
   * @version 1.2
   * @since Version 1.0
   **/

public class receiverStub extends stub implements receiverStubInterface {

    /** receiverStub constructor.
     * <HR>
     * it initialize the stub, registers it as a remote server, and connect it to the core of the multi-channel.
     * <HR>   
     * @param myReferent a representant of the stub referent.
     * @param myKind the multi-channel's name.
     * @param myReferentName the referent's name.
     * @exception ReceiverStubCannotBeRegisteredAsAServerException thrown when the stub cannot be registered as a server.
     * @exception CoreNotFoundException thrown when the core of the multi-channel, which the stub is part of,
     *            doesn't exist or isn't correctly registered as a server. 
     **/

  public receiverStub(Object myReferent, String myKind, String myReferentName) throws RemoteException, ReceiverStubCannotBeRegisteredAsAServerException, CoreNotFoundException {
    setReferent(myReferent); 
    try {
      setWhoIsMyCore(myKind);
      Naming.rebind(myKind+"__"+myReferentName, this); // register myself 
    } catch(MalformedURLException e) { 
        throw new ReceiverStubCannotBeRegisteredAsAServerException("*** Troubles in registering the receiver's stub "+myReferentName+" as a RMI server"); 
    }
  }

    /** 
     * it performs the meta-computation on the receiver side -- calling the method {@link #receiverSideMetaBehavior <B>receiverSideMetaBehavior</B>} --, 
     * then really invoke the method reified by the meta-computationi -- invoking the {@link #tInvoke <B>tInvoke</B>} method.
     * <HR>
     * It is called by the <B>coreMetaBehavior</B> method of the multi-channel core. It cannot be overrode.<BR> 
     * Note that as side-effect the <B>receiverSideMetaBehavior</B> method can change the value of the arguments used by <B>tInvoke</B>.
     * <HR>
     * @param msg the hijacked method call.
     * @return the result of the method invocation
     * @exception MethodDoesNotExistException thrown when the method reified doesn't exist in the referent class. 
     **/

  final public Object invoke(mChaRMMethodCall msg) throws MethodDoesNotExistException {
    beforeReceiverSideMetaBehavior(msg);
    Object res = tInvoke(msg);
    msg.setReturnValue(res);
    afterReceiverSideMetaBehavior(msg);
    return msg.getReturnValue();
  }

  final private Class[] extractArgsClasses(Object[] args) {
    if (args == null) return null; 
    Class[] c = new Class[args.length];
    for (int i=0; i<args.length; i++) c[i] = args[i].getClass();
    return c;
  }
 
  final private String mapToPrimitive(Class c) {
    if ((c.getName()).compareTo("java.lang.Boolean") == 0) return "boolean"; 
    if ((c.getName()).compareTo("java.lang.Character") == 0) return "char"; 
    if ((c.getName()).compareTo("java.lang.Byte") == 0) return "byte"; 
    if ((c.getName()).compareTo("java.lang.Short") == 0) return "short"; 
    if ((c.getName()).compareTo("java.lang.Integer") == 0) return "int"; 
    if ((c.getName()).compareTo("java.lang.Long") == 0) return "long"; 
    if ((c.getName()).compareTo("java.lang.Float") == 0) return "float"; 
    if ((c.getName()).compareTo("java.lang.Double") == 0) return "double"; 
    if ((c.getName()).compareTo("java.lang.Void") == 0) return "void";
    return null;
  }
  
  final private boolean sameArguments(Class[] c1, Class[] c2) {
    String primitive;
    if (c1.length != c2.length) return false;
    for(int i=0; i<c1.length; i++) 
      if ( ( (c1[i].getName()).compareTo(c2[i].getName() )  ) != 0 ) {
        primitive = mapToPrimitive(c2[i]);
        if (primitive == null) return false;
        else if ( ( (c1[i].getName()).compareTo(primitive) ) != 0 ) return false; 
      }
    return true;
  }

    /** it invokes a method with the actual arguments on the receiver side.
     * <HR>
     * it uses the reflection mechanism offered by the Java API Core Reflection. 
     * It is able to handle overloaded method. It cannot neither be overrode nor directly invoked by user.
     * <HR>
     * @param msg the hijacked method call.
     * @return the result of the method invocation.
     * @exception MethodDoesNotExistException thrown when the method reified doesn't exist in the referent class. 
     **/
  
  final protected Object tInvoke(mChaRMMethodCall msg) throws MethodDoesNotExistException {
    Class[] ctemp, c = extractArgsClasses(msg.actualArguments());
    String error = "*** The method: "+msg.getMethodName()+"( ";
    for(int i=0;i<c.length;i++) error += c[i].getName()+" ";
    error += ") wasn't declared in the referent class ["+referent().getClass().getName()+"]";    
    try {
      Method[] methods = (referent()).getClass().getDeclaredMethods();
      for(int i=0; i<methods.length; i++)
        if ((methods[i].getName()).compareTo(msg.getMethodName()) == 0)
          if (sameArguments(methods[i].getParameterTypes(), c))
            return methods[i].invoke(referent(), msg.actualArguments()); 
      throw new MethodDoesNotExistException(error);
    } catch(Exception e) {
      throw new MethodDoesNotExistException(error);
    }
  }  

    /** it is called before the true invocation and performs meta-computation on the message in the target locus.
     * <HR>
     * as default nothing it is done, it is possible to define new
     * behavior defing a new subclass overriding this method.
     * <HR>
     * @param msg the hijacked method call.
     **/

   public void beforeReceiverSideMetaBehavior(mChaRMMethodCall msg) {}

   /** it is called after the true invocation and performs meta-computation on the return value in the target locus.
     * <HR>
     * as default nothing it is done, it is possible to define new
     * behavior defing a new subclass overriding this method.
     * <HR>
     * @param msg the hijacked method call.
     **/
   public void afterReceiverSideMetaBehavior(mChaRMMethodCall msg) {}

}

