package mChaRM.mChaRMCollection;

import java.io.*;
import java.lang.*;
import java.util.*;
import uka.karmi.rmi.*;
import uka.karmi.rmi.server.*;
import java.net.MalformedURLException;
import mChaRM.*;
import mChaRM.multichannel.*;
import mChaRM.RMI.*;


  /**
   * Instances of this class are used as stubs of the multi-channel on the receiver site.
   * Each receiver connected to a multi-channel will be extended by a stub of such a multi-channel.<BR> 
   * This class defines the meta-computations that have to be performed on the receiver side. 
   * @author Michele Ferraro (<A HREF="mailto:1996s106@educ.disi.unige.it">1996s106@educ.disi.unige.it</A>)<br>
   *         Luigi Tosetto (<A HREF="mailto:1996s028@educ.disi.unige.it">1996s028@educ.disi.unige.it</A>)<br>
   *         Davide Zerbino (<A HREF="mailto:1996s027@educ.disi.unige.it">1996s027@educ.disi.unige.it</A>)
   * @version 1.0
   * @since Version 1.2
   **/

public class RMPReceiverStub extends receiverStub implements RMPReceiverStubInterface {

  /** This Vector stores, for each messages arrived, the acknowledge received from others receivers.
   * @serial
   **/
  private Vector otherAcks;
 
  /** This BitSet stores which messages are arrived to this receiver.
   * @serial 
   **/
  private BitSet myAcks;
 
  /** String representing the name of the receiver.
   * @serial
   **/
  private String whoAmI;
 
  /** RMPChannelInterface needed to send a NACK to the RMP Channel.
   * @serial 
   **/
  private RMPChannelInterface cc;
 
  /** String representing the name of the channel
   * @serial 
   **/
  private String serverName;
 
  /** Boolean to set the print option. 
   *If it's true some significant prints are made during the execution, otherwise no print is made.
   * @serial 
   **/
  private static boolean prt = false;

 
  /** RMPReceiverStub constructor.
   * <HR>
   * It calls the constructor of the superclass, stores the name of the receiver, the server and its core,
   * initializes data structures for Acks.
   * <HR>   
   * @param myReferent a representant of the stub referent.
   * @param myKind the multi-channel's name.
   * @param myReferentName the referent's name.
   * @exception ReceiverStubCannotBeRegisteredAsAServerException thrown when the stub cannot be registered as a server.
   * @exception CoreNotFoundException thrown when the core of the multi-channel, which the stub is part of,
   *            doesn't exist or isn't correctly registered as a server.
   * @exception NotBoundException thrown when there is a trouble allocating the space to store acknowledges.
   **/
  
  public RMPReceiverStub (Object myReferent, String myKind, String myReferentName) 
         throws RemoteException, ReceiverStubCannotBeRegisteredAsAServerException, CoreNotFoundException, NotBoundException {
    super(myReferent, myKind, myReferentName);
    whoAmI = myReferentName;
    serverName = myKind;
    otherAcks = new Vector(10);
    myAcks = new BitSet();
    cc = (RMPChannelInterface)WhoIsMyCore();
  }

 
  /** 
   * It adds an Ack for the message n. If the receiver hasn't yet this message it sends to the channel a NAck
   * <HR>
   * When a message arrives to a receiver, pACK is called on each other receiver to notify this event.
   * <HR>
   * @param n the number of the message arrived.
   * @return the string "OK" if it's all right.
   **/

  public String pACK (int n) throws RemoteException {
 
    synchronized (otherAcks) {
      if (n+1 > otherAcks.size()) otherAcks.setSize(n+1);
      if (otherAcks.elementAt(n) == null) otherAcks.setElementAt(new Integer(1), n);
      else otherAcks.setElementAt(new Integer(((Integer)otherAcks.get(n)).intValue()+1), n);
      if (prt) System.out.println("*** RMPReceiverStub " + whoAmI + ", I'm in pACK, ACK arrived: " + otherAcks.get(n));
    } 
    if (!myAcks.get(n)) new sendNAck(1000, n).start();
    return new String("Ok");
  }


  /** 
   * It sends an ACK to all others receivers
   * <HR>
   * To send an ACK it must search the process, then a new sendPack thread starts
   * <HR>
   * @param n the number of the message to be acks.
   * @param receivers the string containing receivers' names.
   **/

  private void ackTheMsg (int n, String[] receivers) throws NotBoundException {
    for (int i = 0; i < receivers.length; i++)
      if (receivers[i].compareTo(whoAmI) != 0)
    new sendPAck(1000, n, (RMPReceiverStubInterface)lookupEveryWhere.lookup(serverName + "__" + receivers[i])).start();
  }
  

  /** 
   * It is called before invoking the method and permits to perform the last meta-computation on the receiver
   * <HR>
   * If the message arrives for the first time it stores it and sends Acks to the others receivers,
   * then waits until all previous messages and Acks are arrived. Otherwise nothing is done.
   * <HR>
   * @param msg the message arrived.
   **/
  
  public void beforeReceiverSideMetaBehavior(mChaRMMethodCall msg) {
    try {
      boolean newMsg = false;
      int msgNumber = ((Integer)msg.removeArgument(0)).intValue();
      String[] receivers = msg.receivers();
   
      synchronized (otherAcks) {
        if (msgNumber+1 > otherAcks.size()) otherAcks.setSize(msgNumber+1);
      }
      synchronized (myAcks) {
        if (!myAcks.get(msgNumber)) {
          newMsg = true;
          myAcks.set(msgNumber);
        } 
      }
      if (newMsg) {
        if (prt) System.out.println("*** Stub "+whoAmI+": new msg " + msgNumber);
        ackTheMsg(msgNumber, receivers);
        if (prt) System.out.println("*** Stub "+whoAmI+": msg " + msgNumber + ", ACKs sent");
      } else return;
      for (int i = 0; i <= msgNumber; i++) 
        while ( (otherAcks.elementAt(i) == null) || !myAcks.get(i) ||
                (((Integer)otherAcks.elementAt(i)).intValue() < receivers.length - 1));
    } catch (ReferenceToNonExistentArgumentException e1) {
      System.out.println("*** Troubles accessing message argument.");
    } catch (NotBoundException e2) {
      System.out.println("*** Receiver not found.");
    } catch (Exception e3) {
      System.out.println("*** Exception in beforeReceiverSideMetabehavior");
      e3.printStackTrace();
    }
  }
 
 
 /**
  * This class is used to send NACK of a particular message with a specified delay
  **/
 
  private class sendNAck extends Thread {
   int delay, msgNumber;
  
   /** 
     * sendNAck constructor
     * <HR>
     * If initializes the delay and the number of the message to be sent.
     * <HR>
     * @param delay the waiting time after sending a NACK.
     * @param msgNumber the number of the message.
     **/
   sendNAck (int delay, int msgNumber) {
     this.delay = delay;
     this.msgNumber = msgNumber;
   }
 
    /** 
     * The overridden method run sends the NACK and repeat this operation
     * if the NACK wasn't sent successfully after the interval specified in delay.
     **/
   public void run () {
     String str = null;
     if (prt) System.out.println("*** RMPReceiverStub " + whoAmI + " asks for retransmission");
     try {
       while (true) {
         str = cc.nACK(msgNumber, whoAmI);
         if (str.compareTo("Ok") != 0) this.sleep(delay);
         else { 
           if (prt) System.out.println("*** RMPReceiverStub nACK was successfull");
           break;
         }
       } 
     } catch (RemoteException e) {
        System.out.println("*** Troubles sending nACK: RemoteException");
     } catch (MethodDoesNotExistException e) {
        System.out.println("*** Troubles sending nACK: MethodDoesNotExistException");
     } catch (InterruptedException e) {
        System.out.println("*** Troubles sending nACK: InterruptedException");
     } 
   } 
 }
 
 
 /**
  * This class is used to send ACK of a particular message to a receiver, with a specified delay
  **/
  private class sendPAck extends Thread {
    int delay, msgNumber;
    RMPReceiverStubInterface rec;
 
     /** 
      * sendPAck constructor
      * <HR>
      * If initializes the delay, the number of the message to be sent and the designed receiver.
      * <HR>
      * @param delay the waiting time after sending a NACK.
      * @param msgNumber the number of the message.
      * @param rec the receiver identifier.
      **/
    sendPAck(int delay, int msgNumber, RMPReceiverStubInterface rec) {
      this.delay = delay;
      this.msgNumber = msgNumber;
      this.rec = rec;
    }
 
     /** 
      * The overridden method run sends the ACK and repeat this operation
      * if the ACK wasn't sent successfully after the interval specified in delay.
      **/
    public void run () {
      String str = null;
      if (prt) System.out.println("*** RMPReceiverStub " + whoAmI + " sends ACK");
        try {
          while (true) {
            str = rec.pACK(msgNumber);
            if (str.compareTo("Ok") != 0) this.sleep(delay);
            else {
              if (prt) System.out.println("*** RMPReceiverStub pACK was successfull");
              break;
            }
          } 
        } catch (RemoteException e) {
            System.out.println("*** Troubles sending pACK: RemoteException");
        } catch (InterruptedException e) {
            System.out.println("*** Troubles sending pACK: InterruptedException");
        } 
    } 
 }
}
