package mChaRM.mChaRMCollection;

import java.io.*;
import java.util.*;
import java.lang.*;
import uka.karmi.rmi.*;
import uka.karmi.rmi.server.*;
import java.net.MalformedURLException;
import mChaRM.*;
import mChaRM.multichannel.*;
import mChaRM.RMI.*;

  /**
   * Realizes a multi-channel of kind <I>RMP</I>. Its meta-behavior, together with the ReceiverStub's
   * one, implements the Reliable Multicast Protolcol by Brian Whetten, Todd Montgomery and Simon Kaplan,
   * which provides a totally ordered, reliable, atomic multicast service on top of an unreliable service.<br>
   * It applies a message number to the message and sends it to the receiverStubs.<br>
   * When results are returned from the receiverStubs it uses a voting algorithm to choose which result will be
   * returned to the senderStub.
   * It also allows the receiverStubs to request the retransmission of the message with the nACK method.<br>
   * It uses instances of senderStub class as senderStubs,
   * and instances of RMPReceiverStub as receiverStubs
   * @author Michele Ferraro (<A HREF="mailto:1996s106@educ.disi.unige.it">1996s106@educ.disi.unige.it</A>)<br>
   *         Luigi Tosetto (<A HREF="mailto:1996s028@educ.disi.unige.it">1996s028@educ.disi.unige.it</A>)<br>
   *         Davide Zerbino (<A HREF="mailto:1996s027@educ.disi.unige.it">1996s027@educ.disi.unige.it</A>)
   * @version 1.0
   * @since Version 1.2
   **/

public class RMPChannel extends channelCore implements RMPChannelInterface {
  /**
   * A Vector containing all the messages sent to the channelCore
   * @serial
   **/
  private Vector messages;
  /**
   * Hashtable indexed on (receiverName, messageNumber); each entry contains
   * the result of message <I>messageNumber</I> sent to receiver <I>messageNumber</I>
   * @serial
   **/
  private Hashtable result;
  /**
   * The message number assigned to all the messages
   * @serial
   **/
  private int msgNumber;
  /**
   * This boolean value indicates whether or not to print some interesting messages
   **/
  private static boolean prt = false;
  
  /** 
   * Constructor of RMP-multi-channels.
   * <HR>
   * This constructor calls the superclass' one fixing the kind "RMP" and specifing classes for
   * senders and receivers that must be "senderStub" and "RMPReceiverStub"
   * It initializes data structures for messages, results and,
   * in the end, initializes the counter for ordering messages.
   * @param RsName an Array of Strings representing the name of the receivers, which it is connected to.
   * @exception ReceiverStubNotFoundException thrown when one of the specified receiver wasn't started before the core.
   **/

  public RMPChannel(String[] RsName) throws RemoteException, ReceiverStubNotFoundException {
    super("RMP", RsName, "mChaRM.multichannel.senderStub", "RMP.RMPReceiverStub");
    messages = new Vector(10);
    result = new Hashtable();
    msgNumber = -1;
  }


  /**
   * Sends the message n to the receiver that has requested it
   * <HR>
   * If the message is correctly sent return the string "Ok"
   * <HR>
   * @param n number of the message required.
   * @param name the name of the receiver.
   * @return "Ok" if all is right.
   **/

  public String nACK (int n, String name) throws MethodDoesNotExistException, RemoteException {
    if (prt) System.out.println("*** Channel Core: called nACK from "+name+", for msg number " + n);
    ((RMPReceiverStubInterface)receiverStub(name)).invoke((mChaRMMethodCall)messages.elementAt(n));
    return new String("Ok");
  }
 
 
  /**
   * Generates a new ordered message number
   * <HR>
   * This method is synchronized because more than one thread in RMPChannel may ask for a new number
   * at the same time.
   * <HR>
   * @return the new message number. 
   **/
 
  private synchronized int newMsg () {
    msgNumber++;
    messages.setSize(msgNumber+1);
    return msgNumber;
  }


  /** This method embodies the reflective behaviour realized by the RMP-multi-channel.
   * <HR>
   * It performs meta-computations on the reified method call:
   * <CENTER>multiRMI(RsName, methodName, args)</CENTER>
   * the computations performed can't make assumption about where their are performed.<BR>
   * It adds to the message a progressive number then multicasts it to the receivers,
   * collects the results and returns the most probable result (obtained with the voting method)
   * of the computation to the sender.
   * <HR>   
   * Note that receivers can be a subset of the multi-channel's receivers
   * <HR>
   * @param msg the hijacked method call.
   * @return the result of the method call.
   * @exception MethodDoesNotExistException thrown when the method reified doesn't exist in the referent class.
   **/

  public Object coreMetaBehavior(mChaRMMethodCall msg) throws MethodDoesNotExistException, RemoteException {
    int myMsg, i= 0, recNumber = (msg.receivers()).length;
    sendMsg[] t = new sendMsg[recNumber];
    pair[] keys = new pair[recNumber];
    Object retVal;

    myMsg = newMsg();
    msg.insertArgument(0, new Integer(myMsg)); 
    messages.setElementAt(msg, myMsg);
    for (i=0; i<recNumber; i++) {
      keys[i] = new pair(myMsg, i);
      if (prt) System.out.println("*** Channel Core: sent message "+myMsg+" to receiver "+msg.receivers()[i]);
      t[i] = new sendMsg((RMPReceiverStubInterface)receiverStub((msg.receivers())[i]), msg, keys[i]);
      t[i].start();
    }
    for (i=0; i<recNumber; i++)
      try {
       t[i].join();
      } catch (InterruptedException e) {
           e.printStackTrace();
      }
    retVal = voting(keys);
    if (prt) System.out.println("*** Channel Core: message "+myMsg+" return "+retVal);
    return retVal;
  }
 
 
  /**
   * Returns the most probable result.
   * @return the final result. 
   **/
 
  private Object voting(pair[] keys) {
    Hashtable res = new Hashtable();
    Object parTmp, tmp = null;
    Enumeration eKeys;
    int i, parNval, nval = 0;
  
    for (i=0; i<keys.length; i++) {
      if (i == 0) res.put(result.get(keys[i]), new Integer(1));
      else {
        tmp = result.get(keys[i]);
        parTmp = res.get(tmp);
        if (parTmp == null) res.put(tmp, new Integer(1));
        else
        res.put(tmp, new Integer(((Integer)res.get(tmp)).intValue()+1));
      }
    }
    eKeys = res.keys();
    for (i=0; eKeys.hasMoreElements(); i++) {
      if (i == 0) {
        tmp = eKeys.nextElement();
        nval = ((Integer)res.get(tmp)).intValue();
      } else {
        parTmp = eKeys.nextElement();
        parNval = ((Integer)res.get(tmp)).intValue();
        if (parNval > nval) {
          tmp = parTmp;
          nval = parNval;
        }
      }
    }
    return tmp;
  }
 
  /**
   * This class is used as index in the result Hashtable. It's simply a pair with
   * two method (getRec and getMsg) used to access its fields.
   * @author Michele Ferraro (<A HREF="mailto:1996s106@educ.disi.unige.it">1996s106@educ.disi.unige.it</A>)<br>
   *         Luigi Tosetto (<A HREF="mailto:1996s028@educ.disi.unige.it">1996s028@educ.disi.unige.it</A>)<br>
   *         Davide Zerbino (<A HREF="mailto:1996s027@educ.disi.unige.it">1996s027@educ.disi.unige.it</A>)
   * @version 1.0
   * @since Version 1.2
   **/
  private class pair {
    private int msgNumber;
    private int recNumber;
 
    /**
     * Constructor for pair
     * @param msgNumber the number of the message
     * @param recNumber an integer value representing a receiver
     **/
   public pair (int msgNumber, int recNumber) {
     this.msgNumber = msgNumber;
     this.recNumber = recNumber;
   }
   
   /**
    * Reads the value of the field recNumber
    * @return simply the value of the field
    **/
   public int getRec () {
     return recNumber;
   }
   
   /**
    * Reads the value of the field msgNumber
    * @return simply the value of the field
    **/
   public int getMsg () {
     return msgNumber;
   }
 }
 
  /**
   * Sends a message to the specified receiverStub and stores the result into the <I>result</I> Hashtable
   * @author Michele Ferraro (<A HREF="mailto:1996s106@educ.disi.unige.it">1996s106@educ.disi.unige.it</A>)<br>
   *         Luigi Tosetto (<A HREF="mailto:1996s028@educ.disi.unige.it">1996s028@educ.disi.unige.it</A>)<br>
   *         Davide Zerbino (<A HREF="mailto:1996s027@educ.disi.unige.it">1996s027@educ.disi.unige.it</A>)
   * @version 1.0
   * @since Version 1.2
   **/
  private class sendMsg extends Thread {
    /**
     * Contains the receiver which the message must be sent
     **/ 
    private RMPReceiverStubInterface rec;
    /**
     * Contains the number of the message to send
     **/
    private mChaRMMethodCall msg;
    /**
     * Contains the key that will be used to access the <I>result</I> Hashtable
     **/
    private pair key;

    /**
     * Constructor for sendMsg
     * @param rec the receiver which the message must be sent
     * @param msg the number of the message to send
     * @param key the key that will be used to access the <I>result</I> Hashtable
     **/
    public sendMsg(RMPReceiverStubInterface rec, mChaRMMethodCall msg, pair key) {
      this.rec = rec;
      this.msg = msg;
      this.key = key;
    }
  
    /**
     * Automatically called when a new instance of sendMsg is created,
     * sends the message to the receiverStub and waits for the result.
     * When the result arrives it's stored in the <I>result</I> Hashtable
     **/
  
    public void run() {
      Object res = null;
      try {
        res = rec.invoke(msg);
      } catch (Exception e) {
        System.out.println("*** Error in Thread's invoke");
        e.printStackTrace();
        return;
      }
      synchronized (result) {
        result.put(key, res);
      }
      if (prt) System.out.println("*** End thread "+key.getRec()+" msg "+key.getMsg());
    }
  }
}
