/*---------------------------------------
	PreUtils.java
---------------------------------------*/
package mChaRM.PremChaRM;

import openjava.mop.*;
import openjava.syntax.*;
import openjava.ptree.*;
import java.io.*;
import java.util.*;


  /**
   * This class contains lots of utilities for <code>PremChaRM</code> class.
   * It contains only static fields and methods.
   *
   * @author Valentina Cord&igrave;
   * @author Maurizio Mazza
   * @author Roberto Montagna
   **/
public class PreUtils
 {
/*---------------------------------------
 Private Constants
---------------------------------------*/

   /**
    * Constant value for debugging. It represent the Syntax checking level
    * of the debugging phase.
    **/
 public static final int dbgSyntax  = 0;

   /**
    * Constant value for debugging. It represent the OpenClasses level
    * of the debugging phase.
    **/
 public static final int dbgOpenClasses  = 1;

   /**
    * Constant value for debugging. It represent the prepearing class level
    * of the debugging phase.
    **/
 public static final int dbgPrepClasses  = 2;

   /**
    * Constant value for debugging. It represent the level of construction of the
    * methods graph in the debugging phase.
    **/
 public static final int dbgGraphMethod  = 3;

   /**
    * Constant value for debugging. It represent the level of construction of the
    * fields graph in the debugging phase.
    **/
 public static final int dbgGraphField  = 4;

   /**
    * Constant value for debugging. It represent the lowest level of the
    * debugging phase.
    **/
 public static final int dbgUtils  = 5;

   /**
    * Constant value for debugging. It represent the printing-to-file level of the
    * debugging phase.
    **/
 public static final int dbgPrintClasses  = 6;

   /**
    * Constant value for debugging. It represent the expressions evalutation level 
    * of the debugging phase.
    **/
 public static final int dbgPrintExpression  = 7;

   /**
    * Constant value for debugging. It represent the allocation expansion level
    * of the debugging phase.
    **/
 public static final int dbgAllocation  = 8;

   /**
    * Constant value for debugging. It represent the method call expansion level
    * of the debugging phase.
    **/
 public static final int dbgMethodCall  = 9;

   /**
    * Boolean array determining which debug levels have to be considered. If the value
    * is <code>false</code> that level must not be present, othewise it must be.
    * It is used by {@link #printDebug(String,int,int) <B>printDebug</B>} method.
    **/
 public static final boolean debugLevel[] = {false, false, false, false, false, false, false, false, false, false};


/*---------------------------------------
 Private Fields
---------------------------------------*/

   /**
    * This field contains the name of the class instanciating PremChaRM class.
    * At the biginning its value is 'null', but it can be changed by the
    * {@link #setName(String) <B>setName</B>} method.
    **/
 private static String className = null;


/*---------------------------------------
 Method
---------------------------------------*/

   /**
    * This method allow to change the name of the class instanciating PremChaRM class.
    * It only sets private field className to <code>name</code>.
    * <hr>
    * @param name the name of the class instanciating PremChaRM class.
    **/
 public static void setName( String name )
  {
  className = name;
  }

   /**
    * This method read the name of the class instanciating PremChaRM class.
    * <hr>
    * @return the name of the class instanciating PremChaRM class.
    **/
 private static String getName()
  {
  return className;
  }

   /**
    * This method tests if two arrays of String have the same values.
    * <hr>
    * @param m1 first array.
    * @param m2 second array.
    * @return the comparison between parameters.
    **/
 public static boolean equalsParameterType(String[] m1,String[] m2)
  {
  PreUtils.printDebug("Start Method equalsParameter String", PreUtils.dbgUtils, 0);

  for( int i=0; i<m1.length; i++ )
    PreUtils.printDebug("equalsParameter param1 "+m1[i], PreUtils.dbgUtils, 1);
  for( int i=0; i<m2.length; i++ )
    PreUtils.printDebug("equalsParameter param2 "+m2[i], PreUtils.dbgUtils, 1);

  if( m1.length != m2.length )
    return false;

  for(int i=0; i<m1.length; i++ )
    if( !m1[i].equals( m2[i] ) )
      return false;

  return true;
  }


   /**
    * This method tests if two OJMethod objects have the same parameter types.
    * It returns true if methods have the same number of parameters and they have
    * the same types.
    * <hr>
    * @param m1 first OJMethod object.
    * @param m2 second OJMethod object.
    * @return the comparison between methods parameters.
    **/
 public static boolean equalsParameterType(OJMethod m1,OJMethod m2)
  {
  PreUtils.printDebug("Start Method equalsParameter", PreUtils.dbgUtils, 0);

  OJClass[] typeM1 = m1.getParameterTypes();
  OJClass[] typeM2 = m2.getParameterTypes();

  if( typeM1.length != typeM2.length )
    return false;

  for(int i=0; i<typeM1.length; i++ )
    if( !typeM1[i].getName().equals( typeM2[i].getName() ) )
      return false;

  return true;
  }

   /**
    * This method returns all MethodCall expressions present in the given OJMethod
    * object's body. It get all statements of the method's body and examine one by one
    * recoursively to find every methods called by that.To do that it calls on every statement the function
    * {@link #getCalledMethodsByExpression(Expression) <B>getCalledMethodsByExpression</B>}.
    * <hr>
    * @param method starting OJMethod object.
    * @return a Vector of MethodCall expressions found in the method's body.
    **/
 public static void getCalledMethods( OJMethod method, Vector foundMethods, Vector allEnvs )
  {
  PreUtils.printDebug("Start Method getCalledMethods", PreUtils.dbgGraphMethod, 0);
  PreUtils.printDebug("method being examined: ", PreUtils.dbgGraphMethod, 1);

  Vector expr;
  OJClass tempClass;
  ClosedEnvironment localEnv = (ClosedEnvironment)method.getEnvironment();
  try{
    StatementList slBody = method.getSourceCode().getBody();
    for( int i=0; i<slBody.size(); i++ )
      getCalledMethodsByStatement( (Statement)slBody.get(i), foundMethods, allEnvs, localEnv  ) ;
    }
  catch( CannotAlterException e )
    {
    printExceptionText( "Unable to locate method ", e );
    }
  }

   /**
    * This method returns all MethodCall expressions found examining given statement.
    * For each statement to examine check if it contains an expression (and if so
    * calls {@link #getCalledMethodsByExpression(Expression) <B>getCalledMethodsByExpression</B>}) or
    * if it contains a variable initializing (in that case calls * {@link #getCalledMethodsByVariableInitializer(VariableInitializer) <B>getCalledMethodsByVariableInitializer(VariableInitializer)</B>} ).
    * <hr>
    * @param stmnt statement to be examined.
    * @return a Vector of MethodCall expressions found starting from given statement.
    **/
 private static void getCalledMethodsByStatement( Statement stmnt, Vector foundMethods, Vector allEnvs, ClosedEnvironment localEnv )
  {
  PreUtils.printDebug("Start Method getCalledMethodsByStmnt", PreUtils.dbgGraphMethod, 0);
  PreUtils.printDebug("statement being examined: "+stmnt, PreUtils.dbgGraphMethod, 1);

  int i;

  if( stmnt instanceof ExpressionStatement )
    getCalledMethodsByExpression( ((ExpressionStatement)stmnt).getExpression(), foundMethods, allEnvs, localEnv );

  else if( stmnt instanceof Block )
    {
    ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
    StatementList BlockSta = ((Block)stmnt).getStatements();
    if( BlockSta != null )
      for( i=0; i<BlockSta.size(); i++ )
        getCalledMethodsByStatement( BlockSta.get(i), foundMethods, allEnvs, newEnv );
    }

  else if( stmnt instanceof DoWhileStatement )
    {
    ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
    getCalledMethodsByExpression( ((DoWhileStatement)stmnt).getExpression(), foundMethods, allEnvs, localEnv );

    StatementList DoSta = ((DoWhileStatement)stmnt).getStatements();
    if( DoSta != null )
      for( i=0; i<DoSta.size(); i++ )
        getCalledMethodsByStatement( DoSta.get(i), foundMethods, allEnvs, newEnv );
    }
      
  else if( stmnt instanceof ForStatement )
    {
    ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
    ForStatement ForSta = (ForStatement)stmnt;

    VariableDeclarator[] newVar = ForSta.getInitDecls();
    if( newVar != null )
      for( i=0; i<newVar.length ; i++ )
        getCalledMethodsByVariableInitializer( newVar[i].getInitializer(), foundMethods, allEnvs, newEnv );

    ExpressionList ForInit = ForSta.getInit();
    if( ForInit != null )
      for( i=0; i<ForInit.size() ; i++ )
        getCalledMethodsByExpression( ForInit.get(i), foundMethods, allEnvs, newEnv );
 
    getCalledMethodsByExpression( ForSta.getCondition(), foundMethods, allEnvs, newEnv );

    ExpressionList ForIncr = ForSta.getIncrement();
    if( ForIncr != null )
      for( i=0; i<ForIncr.size() ; i++ )
        getCalledMethodsByExpression( ForIncr.get(i), foundMethods, allEnvs, newEnv );

    StatementList ForBody = ForSta.getStatements();
    if( ForBody != null )
      for( i=0; i<ForBody.size(); i++ )
        getCalledMethodsByStatement( ForBody.get(i), foundMethods, allEnvs, newEnv );
    }
      
  else if( stmnt instanceof IfStatement )
    {
    ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
    getCalledMethodsByExpression( ((IfStatement)stmnt).getExpression(), foundMethods, allEnvs, newEnv );

    StatementList ThenSta = ((IfStatement)stmnt).getStatements();
    if( ThenSta != null )
      for( i=0; i<ThenSta.size(); i++ )
        getCalledMethodsByStatement( ThenSta.get(i), foundMethods, allEnvs, newEnv );

    StatementList ElseSta = ((IfStatement)stmnt).getElseStatements();
    if( ElseSta != null )
      for( i=0; i<ElseSta.size(); i++ )
        getCalledMethodsByStatement( ElseSta.get(i), foundMethods, allEnvs, newEnv );
    }
      
  else if( stmnt instanceof LabeledStatement )
    getCalledMethodsByStatement( ((LabeledStatement)stmnt).getStatement(), foundMethods, allEnvs, localEnv );
      
  else if( stmnt instanceof SwitchStatement )
    {
    getCalledMethodsByExpression( ((SwitchStatement)stmnt).getExpression(), foundMethods, allEnvs, localEnv );

    CaseGroupList CaseSta = ((SwitchStatement)stmnt).getCaseGroupList();
    if( CaseSta != null )
      for( i=0; i<CaseSta.size(); i++ )
        {
        ExpressionList tempEL = CaseSta.get(i).getLabels();
        if( tempEL != null )
          for( int j=0; j<tempEL.size(); j++ )
	    getCalledMethodsByExpression( tempEL.get(j), foundMethods, allEnvs, localEnv );

        StatementList tempSL = CaseSta.get(i).getStatements();
	ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
        if( tempSL != null )
          for( int j=0; j<tempSL.size(); j++ )
	    getCalledMethodsByStatement( tempSL.get(j), foundMethods, allEnvs, newEnv );
        }
    }
      
  else if( stmnt instanceof ThrowStatement )
    getCalledMethodsByExpression( ((ThrowStatement)stmnt).getExpression(), foundMethods, allEnvs, localEnv );
      
  else if( stmnt instanceof TryStatement )
    {
    ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
    StatementList TrySta = ((TryStatement)stmnt).getBody();
    if( TrySta != null )
      for( i=0; i<TrySta.size(); i++ )
        getCalledMethodsByStatement( TrySta.get(i), foundMethods, allEnvs, newEnv );

    CatchList catchList = ((TryStatement)stmnt).getCatchList();
    if( catchList != null )
      for( i=0; i<catchList.size(); i++ )
        {
        newEnv = new ClosedEnvironment( localEnv );
        StatementList CatchSta = catchList.get(i).getBody();
        if( CatchSta != null )
          for( i=0; i<CatchSta.size(); i++ )
            getCalledMethodsByStatement( CatchSta.get(i), foundMethods, allEnvs, newEnv );
        }

    newEnv = new ClosedEnvironment( localEnv );
    StatementList FinSta = ((TryStatement)stmnt).getFinallyBody();
    if( FinSta != null )
      for( i=0; i<FinSta.size(); i++ )
        getCalledMethodsByStatement( FinSta.get(i), foundMethods, allEnvs, newEnv );
    }
      
  else if( stmnt instanceof WhileStatement )
    {
    ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
    getCalledMethodsByExpression( ((WhileStatement)stmnt).getExpression(), foundMethods, allEnvs, newEnv );

    StatementList WhiSta = ((WhileStatement)stmnt).getStatements();
    if( WhiSta != null )
      for( i=0; i<WhiSta.size(); i++ )
        getCalledMethodsByStatement( WhiSta.get(i), foundMethods, allEnvs, newEnv );
    }
      
  else if( stmnt instanceof ReturnStatement )
    getCalledMethodsByExpression( ((ReturnStatement)stmnt).getExpression(), foundMethods, allEnvs, localEnv);
      
  else if( stmnt instanceof SynchronizedStatement )
    {
    ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
    getCalledMethodsByExpression( ((SynchronizedStatement)stmnt).getExpression(), foundMethods, allEnvs, localEnv);

    StatementList SynSta = ((SynchronizedStatement)stmnt).getStatements();
    if( SynSta != null )
      for( i=0; i<SynSta.size(); i++ )
        getCalledMethodsByStatement( SynSta.get(i), foundMethods, allEnvs, newEnv );
    }

  else if( stmnt instanceof VariableDeclaration )
    {
    VariableDeclaration DecSta = (VariableDeclaration)stmnt;

    try{
      String nameOfType = DecSta.getTypeSpecifier().getName();
      for( i=0; i<DecSta.getTypeSpecifier().getDimension(); i++ )
        nameOfType += "[]";
      localEnv.bindVariable( DecSta.getVariable(), OJClass.forName( nameOfType ) );
      }
    catch( OJClassNotFoundException exc )
      {
      printExceptionText( "FATAL ERROR: unexpected error while looking for a class", exc );
      }

    getCalledMethodsByVariableInitializer( DecSta.getInitializer(), foundMethods, allEnvs, localEnv );

    if( DecSta.getVariableDeclarator() != null )
      getCalledMethodsByVariableInitializer( DecSta.getVariableDeclarator().getInitializer(), foundMethods, allEnvs, localEnv );
    }

  }

   /**
    * This method returns all MethodCall expressions found examining given VariableInitializer.
    * For each VariableInitializer checks if it is an instance of expression (and if so calls 
    * {@link #getCalledMethodsByExpression(Expression) <B>getCalledMethodByExpression</B>}
    * If instead it is an instance of ArrayInitializer it calls recoursively thi method for each
    * element of the array.
    * <hr>
    * @param varInit VariableInitializar object to be examined.
    * @return  Vector of MethodCall expressions found starting from given VariableInitializar object.
    **/
 private static void getCalledMethodsByVariableInitializer( VariableInitializer varInit, Vector foundMethods, Vector allEnvs, ClosedEnvironment localEnv )
  {
  PreUtils.printDebug("Start Method getCalledMethodsByVarInit", PreUtils.dbgGraphMethod, 0);
  PreUtils.printDebug("var. init. being examined: "+varInit, PreUtils.dbgGraphMethod, 1);

  int i;

  if( varInit instanceof Expression )
    getCalledMethodsByExpression( (Expression)varInit, foundMethods, allEnvs, localEnv );

  else if( varInit instanceof ArrayInitializer )
    {
    ArrayInitializer arrayInit = (ArrayInitializer )varInit;
    for(i=0; i<arrayInit.size(); i++)
      getCalledMethodsByVariableInitializer( arrayInit.get(i), foundMethods, allEnvs, localEnv );
    }
  }


   /**
    * This method returns all MethodCall expressions found examining given expression.
    * If given expression in an instance of MethodCall this method adds it to the Vector
    * of founded MethodCall. Then it examines all sub-expression recousively.
    * <hr>
    * @param expr Expression object to be examined.
    * @return  Vector of MethodCall expressions found starting from given expression.
    **/
 private static void getCalledMethodsByExpression( Expression expr, Vector foundMethods, Vector allEnvs, ClosedEnvironment localEnv )
  {
  PreUtils.printDebug("Start Method getCalledMethodsByExpr", PreUtils.dbgGraphMethod, 0);
  PreUtils.printDebug("expression being examined: "+expr, PreUtils.dbgGraphMethod, 1);

  int i;

  if( expr instanceof MethodCall )
    {
    PreUtils.printDebug("I'm a MethodCall", PreUtils.dbgPrintExpression, 2);
    ClosedEnvironment newEnv = new ClosedEnvironment( localEnv );
    foundMethods.add( expr );
    allEnvs.add( newEnv );

    Expression refExpr = ((MethodCall)expr).getReferenceExpr();
    TypeName refType = ((MethodCall)expr).getReferenceType();
    if( refExpr == null && refType != null && refType.getName().equals( getName() ) )
      ((MethodCall)expr).setReferenceType( null );

    getCalledMethodsByExpression( ((MethodCall)expr).getReferenceExpr(), foundMethods, allEnvs, localEnv );

    PreUtils.printDebug("I'm a ReferenceType "+((MethodCall)expr).getReferenceType(), PreUtils.dbgPrintExpression, 5);

    for( i = 0; i < ((MethodCall)expr).getArguments().size() ; i++ )
      getCalledMethodsByExpression( ((MethodCall)expr).getArguments().get(i), foundMethods, allEnvs, localEnv );
    }

  else if( expr instanceof AllocationExpression )
    {
    PreUtils.printDebug("I'm an AllocationExpression", PreUtils.dbgPrintExpression, 2);
    AllocationExpression AllExpr = ((AllocationExpression)expr); 
    if( AllExpr.getArguments() != null )
      for( i = 0; i < AllExpr.getArguments().size() ; i++ )
        getCalledMethodsByExpression( AllExpr.getArguments().get(i), foundMethods, allEnvs, localEnv );

    getCalledMethodsByExpression( AllExpr.getEncloser(), foundMethods, allEnvs, localEnv );
    /*-------------------------------------------
       Member Declaration List not Supported
    -------------------------------------------*/
    }
	
  else if( expr instanceof ArrayAccess )  
    {
    PreUtils.printDebug("I'm an ArrayAccess", PreUtils.dbgPrintExpression, 2);
    getCalledMethodsByExpression( ((ArrayAccess)expr).getIndexExpr(), foundMethods, allEnvs, localEnv );
    getCalledMethodsByExpression( ((ArrayAccess)expr).getReferenceExpr(), foundMethods, allEnvs, localEnv );
    }

  else if( expr instanceof ArrayAllocationExpression )
    {
    PreUtils.printDebug("I'm an ArrayAllocationExpression", PreUtils.dbgPrintExpression, 2);
    ArrayAllocationExpression arrayExpr = (ArrayAllocationExpression)expr;

    ExpressionList listExpr = arrayExpr.getDimExprList();
    if( listExpr != null )
      for( i = 0; i < listExpr.size() ; i++ )
        getCalledMethodsByExpression( listExpr.get(i), foundMethods, allEnvs, localEnv );

    ArrayInitializer arrayIniz = arrayExpr.getInitializer();
    if( arrayIniz != null )
      for( i = 0; i < arrayIniz.size() ; i++ )
        getCalledMethodsByVariableInitializer( (Expression)arrayIniz.get(i), foundMethods, allEnvs, localEnv );
    }

  else if( expr instanceof AssignmentExpression )
    {
    PreUtils.printDebug("I'm an AssignmentExpression", PreUtils.dbgPrintExpression, 2);
    getCalledMethodsByExpression( ((AssignmentExpression)expr).getLeft(), foundMethods, allEnvs, localEnv );
    getCalledMethodsByExpression( ((AssignmentExpression)expr).getRight(), foundMethods, allEnvs, localEnv );
    }

  else if( expr instanceof BinaryExpression )
    {
    PreUtils.printDebug("I'm a BinaryExpression", PreUtils.dbgPrintExpression, 2);
    getCalledMethodsByExpression( ((BinaryExpression)expr).getLeft(), foundMethods, allEnvs, localEnv );
    getCalledMethodsByExpression( ((BinaryExpression)expr).getRight(), foundMethods, allEnvs, localEnv );
    }

  else if(  expr instanceof CastExpression )  
    {
    PreUtils.printDebug("I'm a CastExpression", PreUtils.dbgPrintExpression, 2);
    getCalledMethodsByExpression( ((CastExpression)expr).getExpression(), foundMethods, allEnvs, localEnv );
    }

  else if( expr instanceof ConditionalExpression )
    {
    PreUtils.printDebug("I'm a ConditionalExpression", PreUtils.dbgPrintExpression, 2);
    getCalledMethodsByExpression( ((ConditionalExpression)expr).getCondition(), foundMethods, allEnvs, localEnv );
    getCalledMethodsByExpression( ((ConditionalExpression)expr).getTrueCase(), foundMethods, allEnvs, localEnv );
    getCalledMethodsByExpression( ((ConditionalExpression)expr).getFalseCase(), foundMethods, allEnvs, localEnv );
    }

  else if( expr instanceof FieldAccess )  
    {
    Expression refExpr = ((FieldAccess)expr).getReferenceExpr();
    TypeName refType = ((FieldAccess)expr).getReferenceType();
    PreUtils.printDebug("Before: " + refType, PreUtils.dbgPrintExpression, 3);
    if( refExpr == null && refType.getName().equals( getName() ) )
      ((FieldAccess)expr).setReferenceType( null );
    PreUtils.printDebug("After: " + ((FieldAccess)expr).getReferenceType(), PreUtils.dbgPrintExpression, 3);

    PreUtils.printDebug("I'm a FieldAccess", PreUtils.dbgPrintExpression, 2);
    getCalledMethodsByExpression( ((FieldAccess)expr).getReferenceExpr(), foundMethods, allEnvs, localEnv );
    }

  else if( expr instanceof InstanceofExpression ) 
    {
    PreUtils.printDebug("I'm an InstanceofExpression", PreUtils.dbgPrintExpression, 2);
    getCalledMethodsByExpression( ((InstanceofExpression)expr).getExpression(), foundMethods, allEnvs, localEnv );
    }

  else if( expr instanceof UnaryExpression ) 
    {
    PreUtils.printDebug("I'm an UnaryExpression", PreUtils.dbgPrintExpression, 2);
    getCalledMethodsByExpression( ((UnaryExpression)expr).getExpression(), foundMethods, allEnvs, localEnv );
    }
  }

   /**
    * This method returns all fields accessed by the given statements. To do that it
    * examins every single statement (using {@link #statementHandling(Statement,Vector,Vector) <B>statementHandling</B>} method)
    * to find used fields.
    * <hr>
    * @param body starting statements to check (usually it is the body of a method).
    * @param parameters a String Vector containing starting variables (usually these are a method's parameters).
    * @param ojFields fields already declared (usually got with getAllFields).
    * @return Vector of OJFields accessed by the given statements.
    **/
 public static Vector fieldHandling( StatementList body, Vector parameters, Vector ojFields)
  {
  PreUtils.printDebug("Start Method fieldHandling", PreUtils.dbgGraphField, 0);

  Vector variables = new Vector();
  if( body != null )
    for( int i=0; i<body.size(); i++ )
      variables.addAll( statementHandling(body.get(i), parameters, ojFields) );
  return variables;
  }

   /**
    * This method returns all fields accessed by the given VariableInitializer.
    * For each VariableInitializer checks if it is an instance of expression and then find all fields accessed
    * by this expression calling {@link #expressionHandling(Expression,Vector,Vector) <B>expressionHandling</B>}
    * If instead it is an instance of ArrayInitializer it calls recoursively thi method for each
    * element of the array.
    * <hr>
    * @param varInit starting VariableInitializer object to check.
    * @param fields a String Vector containing starting variables.
    * @param ojFields fields already declared.
    * @return Vector of OJFields accessed by the given VariableInitializer.
    **/
 private static Vector VariableInitializerHandling( VariableInitializer varInit, Vector fields, Vector ojFields )
  {
  PreUtils.printDebug("Start Method VarInitHandling", PreUtils.dbgGraphField, 0);
  PreUtils.printDebug("var. int. being examined "+varInit, PreUtils.dbgGraphField, 1);

  Vector variables = new Vector();
  int i;

  if( varInit instanceof Expression )
    variables.addAll( expressionHandling( (Expression)varInit, fields, ojFields ) );

  else if( varInit instanceof ArrayInitializer )
    {
    ArrayInitializer arrayInit = (ArrayInitializer)varInit;
    for(i=0; i<arrayInit.size(); i++)
      variables.addAll( VariableInitializerHandling( arrayInit.get(i), fields, ojFields ) );
    }
  
  return variables;
  }

   /**
    * This method returns all fields accessed by the given statement.
    * It make a recoursive search starting from given statement examining
    * each other statement inside it.  When examining the statement
    * a copy of the fields Vector is passed (using <code>clone()</code> method). So new local
    * declared variables are stored into the copy and their scope is local to the statement.
    * <hr>
    * @param stmnt starting statement to check.
    * @param fields a String Vector containing starting variables.
    * @param ojFields fields already declared.
    * @return Vector of OJFields accessed by the given statement.
    **/
 private static Vector statementHandling( Statement stmnt, Vector fields, Vector ojFields)
  {
  PreUtils.printDebug("Start Method stmntHandling", PreUtils.dbgGraphField, 0);

  ExpressionList tempExpr;
  Vector variables = new Vector();

  PreUtils.printDebug("statement being examined: "+stmnt, PreUtils.dbgGraphField, 1);
  int i;

  if( stmnt instanceof VariableDeclaration )
    {
    VariableDeclaration varDecl = (VariableDeclaration)stmnt;

    fields.add( varDecl.getVariable() );
    variables.addAll( VariableInitializerHandling( varDecl.getInitializer(), fields, ojFields) );

    if( varDecl.getVariableDeclarator() != null )
      variables.addAll( VariableInitializerHandling( varDecl.getVariableDeclarator().getInitializer(), fields, ojFields ) );
    }
  
  else if( stmnt instanceof Block )
    variables.addAll( fieldHandling( ((Block)stmnt).getStatements(), (Vector)fields.clone(), ojFields) );
  
  else if( stmnt instanceof DoWhileStatement )
    {
    variables.addAll( fieldHandling( ((DoWhileStatement)stmnt).getStatements(), (Vector)fields.clone(), ojFields) );
    variables.addAll( expressionHandling( ((DoWhileStatement)stmnt).getExpression(), (Vector)fields.clone(), ojFields ) );
    }

  else if( stmnt instanceof ExpressionStatement )
    variables.addAll( expressionHandling( ((ExpressionStatement)stmnt).getExpression(), (Vector)fields.clone(), ojFields ) );
     
  else if( stmnt instanceof ForStatement )
    {
    VariableDeclarator[] newVar = ((ForStatement)stmnt).getInitDecls();
    Vector localFor = (Vector)fields.clone();

    if( newVar != null )
      for( i=0; i<newVar.length; i++ )
        {
        localFor.add( newVar[i].getVariable() );
	variables.addAll( VariableInitializerHandling( newVar[i].getInitializer(), fields, ojFields ) );
        } 
    tempExpr = ((ForStatement)stmnt).getInit(); 
    if( tempExpr != null )
      for( i=0; i<tempExpr.size(); i++ )
        variables.addAll( expressionHandling( tempExpr.get(i), (Vector)localFor.clone(), ojFields ) );

    variables.addAll( expressionHandling( ((ForStatement)stmnt).getCondition(), (Vector)localFor.clone(), ojFields ) );
    variables.addAll( fieldHandling( ((ForStatement)stmnt).getStatements(), (Vector)localFor.clone(), ojFields) );
    tempExpr = ((ForStatement)stmnt).getIncrement(); 
    if( tempExpr != null )
      for( i=0; i<tempExpr.size(); i++ )
	variables.addAll( expressionHandling( tempExpr.get(i), (Vector)localFor.clone(), ojFields ) );
    }

  else if( stmnt instanceof IfStatement )
    {
    variables.addAll( fieldHandling( ((IfStatement)stmnt).getStatements(), (Vector)fields.clone(), ojFields) );
    variables.addAll( fieldHandling( ((IfStatement)stmnt).getElseStatements(), (Vector)fields.clone(), ojFields) );
    variables.addAll( expressionHandling( ((IfStatement)stmnt).getExpression(), (Vector)fields.clone(), ojFields ) );
    }

  else if( stmnt instanceof LabeledStatement )
    variables.addAll( statementHandling( ((LabeledStatement)stmnt).getStatement(), (Vector)fields.clone(), ojFields) );

  else if( stmnt instanceof SwitchStatement )
    {
    variables.addAll( expressionHandling( ((SwitchStatement)stmnt).getExpression(), (Vector)fields.clone(), ojFields ) );

    CaseGroupList arrayCase = ((SwitchStatement)stmnt).getCaseGroupList();
    if( arrayCase != null )
      for( i=0; i<arrayCase.size() ; i++ )
	{
	variables.addAll( fieldHandling( arrayCase.get(i).getStatements(), (Vector)fields.clone(), ojFields) );
	tempExpr = arrayCase.get(i).getLabels();
	if( tempExpr != null )
	  for( i=0; i<tempExpr.size(); i++ )
	    variables.addAll( expressionHandling( tempExpr.get(i), (Vector)fields.clone(), ojFields ) );
	}
    }

  else if( stmnt instanceof ThrowStatement )
    variables.addAll( expressionHandling( ((ThrowStatement)stmnt).getExpression(), (Vector)fields.clone(), ojFields ) );
    
  else if( stmnt instanceof TryStatement )
    {
    variables.addAll( fieldHandling( ((TryStatement)stmnt).getBody(), (Vector)fields.clone(), ojFields) );
    variables.addAll( fieldHandling( ((TryStatement)stmnt).getFinallyBody(), (Vector)fields.clone(), ojFields) );

    CatchList catchList = ((TryStatement)stmnt).getCatchList();
    if( catchList != null )
      for( i=0; i<catchList.size(); i++ )
	variables.addAll( fieldHandling( catchList.get(i).getBody(), (Vector)fields.clone(), ojFields ) );
    }

  else if( stmnt instanceof WhileStatement )
    {
    variables.addAll( fieldHandling( ((WhileStatement)stmnt).getStatements(), (Vector)fields.clone(), ojFields) );
    variables.addAll( expressionHandling( ((WhileStatement)stmnt).getExpression(), (Vector)fields.clone(), ojFields ) );
    }

  else if( stmnt instanceof ReturnStatement )
    variables.addAll( expressionHandling( ((ReturnStatement)stmnt).getExpression(), (Vector)fields.clone(), ojFields ) );
    
  else if( stmnt instanceof SynchronizedStatement )
    {
    variables.addAll( fieldHandling( ((SynchronizedStatement)stmnt).getStatements(), (Vector)fields.clone(), ojFields) );
    variables.addAll( expressionHandling( ((SynchronizedStatement)stmnt).getExpression(), (Vector)fields.clone(), ojFields ) );
    }

  return variables;
  }

    /**
     * This method returns all fields accessed by the given expression.
     * It make a recoursive search starting from given expression examining
     * each sub-expresson it has.
     * <hr>
     * @param stmnt starting statement to check.
     * @param fields a String Vector containing starting variables.
     * @param ojFields fields already declared.
     * @return Vector of OJFields accessed by the given expression.
     **/
 private static Vector expressionHandling( Expression expr, Vector fields, Vector ojFields )
  {
  PreUtils.printDebug("Start Method exprHandling", PreUtils.dbgGraphField, 0);
  int i;
  ExpressionList tempExpr;
  Vector variables = new Vector();
  PreUtils.printDebug("expression being examined "+expr, PreUtils.dbgGraphField, 1);

  if( expr instanceof AllocationExpression )
    {

    AllocationExpression AllExpr = ((AllocationExpression)expr); 
    if( AllExpr.getArguments() != null )
      for( i = 0; i < AllExpr.getArguments().size() ; i++ )
	variables.addAll( expressionHandling( AllExpr.getArguments().get(i), (Vector)fields.clone(), ojFields) );

    variables.addAll( expressionHandling( AllExpr.getEncloser(), (Vector)fields.clone(), ojFields) );

    /*-------------------------------------------
       Member Declaration List not Supported
    -------------------------------------------*/
    }
   
  else if( expr instanceof ArrayAccess )
    {
    variables.addAll( expressionHandling( ((ArrayAccess)expr).getIndexExpr(), (Vector) fields.clone(), ojFields ) );
    variables.addAll( expressionHandling( ((ArrayAccess)expr).getReferenceExpr(), (Vector) fields.clone(), ojFields ) );
    }
  
  else if( expr instanceof ArrayAllocationExpression )
    {
    ArrayAllocationExpression arrayExpr = (ArrayAllocationExpression)expr;

    ExpressionList listExpr = arrayExpr.getDimExprList();
    if( listExpr != null )
      for( i = 0; i < listExpr.size() ; i++ )
	variables.addAll( expressionHandling( listExpr.get(i), (Vector) fields.clone(), ojFields) );

    ArrayInitializer arrayIniz = arrayExpr.getInitializer();
    if( arrayIniz != null )
      for( i = 0; i < arrayIniz.size() ; i++ )
        variables.addAll ( VariableInitializerHandling( arrayIniz.get(i), (Vector) fields.clone(), ojFields) );
    }

  else if( expr instanceof AssignmentExpression )
    {
    variables.addAll( expressionHandling( ((AssignmentExpression)expr).getLeft(), (Vector) fields.clone(), ojFields ) ); 
    variables.addAll( expressionHandling( ((AssignmentExpression)expr).getRight(), (Vector) fields.clone(), ojFields ) ); 
    }
  
  else if( expr instanceof BinaryExpression )
    {
    variables.addAll( expressionHandling( ((BinaryExpression)expr).getLeft(), (Vector) fields.clone(), ojFields ) ); 
    variables.addAll( expressionHandling( ((BinaryExpression)expr).getRight(), (Vector) fields.clone(), ojFields ) ); 
    }

  else if( expr instanceof CastExpression )
    variables.addAll( expressionHandling( ((CastExpression)expr).getExpression(), (Vector) fields.clone(), ojFields ) ); 

  else if( expr instanceof ConditionalExpression )
    {
    variables.addAll( expressionHandling( ((ConditionalExpression)expr).getCondition(), (Vector) fields.clone(), ojFields ) ); 
    variables.addAll( expressionHandling( ((ConditionalExpression)expr).getFalseCase(), (Vector) fields.clone(), ojFields ) ); 
    variables.addAll( expressionHandling( ((ConditionalExpression)expr).getTrueCase(), (Vector) fields.clone(), ojFields ) ); 
    }
    
  else if( expr instanceof FieldAccess )
    variables.addAll( expressionHandling( ((FieldAccess)expr).getReferenceExpr(), (Vector) fields.clone(), ojFields ) ); 

  else if( expr instanceof InstanceofExpression )
    variables.addAll( expressionHandling( ((InstanceofExpression)expr).getExpression(), (Vector) fields.clone(), ojFields ) ); 

  else if( expr instanceof MethodCall )
    {
    variables.addAll( expressionHandling( ((MethodCall)expr).getReferenceExpr(), (Vector) fields.clone(), ojFields ) ); 
    tempExpr=((MethodCall)expr).getArguments();

    if( tempExpr != null )
      for( i=0; i<tempExpr.size(); i++ )
	variables.addAll( expressionHandling( tempExpr.get(i), (Vector) fields.clone(), ojFields) );
    }
   
  else if( expr instanceof UnaryExpression )
    variables.addAll( expressionHandling( ((UnaryExpression)expr).getExpression(), (Vector) fields.clone(), ojFields ) ); 
  
  else if( expr instanceof Variable )
    {
    for( i=0; i<fields.size(); i++ )
      if( ((Variable)expr).toString().equals( ((String)fields.elementAt(i)) ) )
        {
        PreUtils.printDebug("local access, no problem " + (Variable)expr, PreUtils.dbgGraphField, 1);
        return variables;
        }
    for( i=0; i<ojFields.size(); i++ )
      if( ((Variable)expr).toString().equals( ((OJField)ojFields.elementAt(i)).getName() ) )
        {
        PreUtils.printDebug("multiple reference, dangerous " + (Variable)expr, PreUtils.dbgGraphField, 1);
        variables.add( ojFields.elementAt(i) );
        }
    }
  return variables;

  }

   /**
    * This utility get a Vector object from an array one.
    * <hr>
    * @param inputArray input Object array to be changed into Vector.
    * @return Vector corresponding to given array.
    **/
 public static Vector array2Vector( Object[] inputArray )
  {
  PreUtils.printDebug("Start Method array2Vector", PreUtils.dbgUtils, 0);
  if( inputArray == null ) 
    return new Vector();
  Vector output = new Vector(inputArray.length); 
  for( int i=0; i<inputArray.length; i++ )
    output.add( i, inputArray[i] ); 
  return output;
  }

   /**
    * Check if two given OJMethod objects are overridden. To do that
    * it check if they have the same name and the same parameters
    * (using {@link #equalsParameterType(OJMethod,OJMethod) <B>equalsParameterType</B>}
    * <hr>
    * @param ojMethod1 first OJMethod object.
    * @param ojMethod2 second OJMethod object.
    * @return true if given methods are overridden.
    **/
 static boolean overriddenMethod( OJMethod ojMethod1, OJMethod ojMethod2 )
  {
  PreUtils.printDebug("Start Method overriddenMethod", PreUtils.dbgUtils, 0);
  if( ojMethod1.getName().equals(ojMethod2.getName() )&&  equalsParameterType( ojMethod1, ojMethod2 ) ) 
    return true;
  return false;
  }

   /**
    * Print on standard error the given string followed by the stack trace of
    * the given expression.
    * <hr>
    * @param Text String to be printed.
    * @param e Exception whose stack trace must be printed.
    **/
 public static void printExceptionText( String Text, Throwable e )
  {
  System.err.println( Text + e.getMessage() );
  e.printStackTrace();
  } 

   /**
    * Print on standard output the given string if the cell of debugLevel[] specified by type is true.
    * <hr>
    * @param Msg String to be printed.
    * @param type index of debudLevel[] rappresenting the type of print. 
    * @param level rappresent the indent of print.
    **/
 public static void printDebug( String Msg, int type, int level )
  {
  if( ! debugLevel[type] )
    return;

  for( int i=0; i<level; i++ )
    System.out.print( "  " );
  System.out.println( Msg );
  }

 }
