package mChaRM.PremChaRM;

import openjava.mop.*;
import openjava.ptree.*;
import java.io.*;
import java.util.*;

  /**
   * This class represents the structure of a class and allow to write
   * to disk the code it contains.
   * In particulary it contains methods to create the classes and to get 
   * some info about them.
   * The difference with OJClass is that the last one don't allow to write a
   * code of a new class, but it allow to modify current class of "ojc" 
   * compilation, OJSubClass permit to create a new Java class.

   * @author Valentina Cord&igrave; (<A HREF="mailto:1996s081@educ.disi.unige.it">1996s081@educ.disi.unige.it</A>)
   * @author Maurizio Mazza (<A HREF="mailto:1996s065@educ.disi.unige.it">1996s065@educ.disi.unige.it</A>)
   * @author Roberto Montagna (<A HREF="mailto:1996s063@educ.disi.unige.it">1996s063@educ.disi.unige.it</A>)
   **/

public class OJSubClass extends OJClass  
 {
/*--------------------------------------
 Public Constants
---------------------------------------*/

/*---------------------------------------
 Private Constants
---------------------------------------*/

/*---------------------------------------
 Private Fields
---------------------------------------*/

  /**
   * This is the name of the class. This name is the same given to created output java file.
   **/
 private String className;

  /**
   * File object representing output class.
   **/
 private PrintStream psOutputFile;

  /**
   * This field contains the Superclass of this OJClass.
   **/
 private OJClass ojcMySuperClass;

  /**
   * This field contains the interface implemented by this class.
   **/
 private String[] ojcImplementInterface;

  /**
   * This field contains all the fields that must be inserted into the class.
   **/
  private Vector ojfMyFields = new Vector();

  /**
   * This field contains all the methods that must be inserted into the output class.
   **/
 private Vector ojmMyMethods = new Vector();

  /**
   * This field contains the constructors.
   * Its value is set by
   * {@link #addNewConstructor <B>addNewConstructor</B>} method.
   **/
 public Vector constructors;

  /**
   * This field contains, if present, the invocation of costructor in first line of
   * constructor.
   */
 public Vector fstLineConstr;

  /**
   * This field contains all the imports of the class.
   * At the biginning its value is 'null', but it can be changed by the
   * {@link #setImports <B>setImports</B>} method.
   **/
 private String[] strImports;


/*--------------------------------------------
 Constructors
--------------------------------------------*/

 /**
  * This constructor have to be used if class have no interfaces.
  *<HR> 
  * @param name name of the class
  * @param extendsname name of class to extend.
  **/
 public OJSubClass(String name, String extendsName, String[] imports) throws ClassNotFoundException
  {
  this(name, extendsName, null, imports);
  }

  /**
   * This constructor have to be used if class have some interfaces.
   *<HR>
   * @param name name of the class.
   * @param extendsname name of class to extend.
   * @param interfaceName name of interface that are implemented by this class.
   **/
 public OJSubClass(String name, String extendsName, String[] interfaceName, String[] imports) throws ClassNotFoundException
  {
  super( Class.forName( "java.lang.Object" ), null );

  boolean errorOccours = false;

  if( imports != null )
    strImports = imports;

  className = name;
  try{
    if( extendsName != null )
      ojcMySuperClass =  OJClass.forName( extendsName );

    ojcImplementInterface = interfaceName;

    psOutputFile = new PrintStream( new FileOutputStream( className + ".java" ) );

    constructors = new Vector();
    fstLineConstr = new Vector();
    }
  catch( OJClassNotFoundException e )
    {
    PreUtils.printExceptionText( "Unable to find class ", e );
    errorOccours = true;
    }
  catch( FileNotFoundException e )
    {
    PreUtils.printExceptionText( "Unable to create File ", e );
    errorOccours = true;
    }
  catch( SecurityException e )
    {
    PreUtils.printExceptionText( "You don't have permission to create File ", e );
    errorOccours = true;
    } 
  finally
    {
    if( errorOccours )
      {
      System.err.println("Fatal error!");
      System.exit(-1);
      }
    }
  } 

/*--------------------------------------------
 Method
--------------------------------------------*/

  /**
   * This method return name of inmpleted interface.
   **/
 public String[] getInterfacesStr()
  {
  return ojcImplementInterface; 
  }

  /**
   * This method return name of this class.
   **/
 public String getName()
  {
  return className;
  }

  /**
   * This method set the imports of this class.
   * @param newImports array of imports to set
   **/
 public void setImports( String[] newImports )
  {
  strImports = newImports;
  }

  /**
   * This method add fields to the class 
   * @param fields vector of fields to add 
   **/
 public void addNewFields( Vector fields )
  {
  ojfMyFields.addAll(fields);
  }

  /**
   * This method add field to the class 
   * @param field field to add 
   **/
 public void addNewFields( OJField field )
  {
  ojfMyFields.add(field);
  }

  /**
   * This method delete field to the class 
   * @param field field to delete 
   **/
 public boolean DeleteFields( OJField field )
  {
  return ojfMyFields.remove(field);
  }

  /**
   * This method delete fields to the class 
   * @param fields vector of field to delete 
   **/
 public boolean DeleteFields( Vector fields )
  {
  return ojfMyFields.removeAll(fields);
  }

  /**
   * Add methods to the class 
   * @param methods vector of methods to add 
   **/
 public void addNewMethods( Vector methods )
  {
  ojmMyMethods.addAll(methods);
  }

  /**
   * Add method to the class 
   * @param method method to add 
   **/
 public void addNewMethods( OJMethod method )
  {
  ojmMyMethods.add(method);
  }

  /**
   * Delete method to the class 
   * @param method method to delete 
   **/
 public boolean DeleteMethods( OJMethod method )
  {
  return ojmMyMethods.remove(method);
  }

  /**
   * Delete methods to the class 
   * @param methods vector of methods to delete 
   **/
 public boolean DeleteMethods( Vector methods )
  {
  return ojmMyMethods.removeAll(methods);
  }

  /**
   * This method add Constructor to the class
   * @param plParams constructor parameters
   * @param strException constructor exception
   **/
 public void addNewConstructor( OJMethod constructor )
  {
  String finish = new String("");

  try{
    if( constructor.getBody().size() != 0 )
      {
      Statement stm = constructor.getBody().get(0);
      if( stm instanceof VariableDeclaration )
        {
        VariableDeclaration varDecl = (VariableDeclaration)stm;
        if( varDecl.getTypeSpecifier().getName().equals( "java.lang.String" ) )
          if( varDecl.getVariable().equals( "__constr" ) )
            if( varDecl.getInitializer() instanceof Literal )
              {
              String start = varDecl.getInitializer().toString();
              for(int z=0; z<start.length(); z++ )
                if( start.charAt(z) == '\\' )
                  if( z<start.length() && start.charAt(z+1)=='"' )
                    finish += new String( new char[] { start.charAt(++z) } );
                   else
                    finish += new String( new char[] { start.charAt(z) } );
                 else
                  finish += new String( new char[] { start.charAt(z) } );
              finish = finish.substring(1, finish.length()-1);
              constructor.getBody().remove(0);
              }
        }
      }
    }
  catch( CannotAlterException e )
    { 
    System.err.println("FATAL ERROR: Unable to manipulate the constructor");
    System.exit(-1);
    } 
  this.constructors.add( constructor );
  this.fstLineConstr.add( finish );
  }

 protected void finalize()
  {
  psOutputFile.close();
  }


  /**
   * This method print the initial part of the class.
   * In particullary it prints imports calling {@link #printImports <B>PrintImports</B>}, then
   * prints the declaration of the class and finally it prints extends clause and implements clause
   **/
 public void printHeader()
  {
  psOutputFile.println( "/*\n This file has been generated by PremChaRM\n*/\n" );

  printImports();
  psOutputFile.println( "" );
  psOutputFile.print( getModifiers() + " " );
  psOutputFile.print( "class " + className );

  if( ojcMySuperClass != null )
    psOutputFile.print( " extends " + ojcMySuperClass.getName() );

  if( ojcImplementInterface != null )
    {
    psOutputFile.print( " implements " );
    for( int i=0; i<ojcImplementInterface.length ;  )
      {
      psOutputFile.print( ojcImplementInterface[i] );
      if( ++i < ojcImplementInterface.length )
	psOutputFile.print( ", " );
      }  
    }
  }    

  /**
   * This method return the superclass of this class.
   **/
 public OJClass getSuperClass()
  {
  return ojcMySuperClass;
  }

  /**
   * This method print all imports int the output file.
   **/
 public void printImports()
  {
  if( strImports != null )
    for( int i=0; i<strImports.length; i++ )
      psOutputFile.println( "import " + strImports[i] + ";" );
  }

  /**
   * This method print the whole class.
   * It calls in order : {@link #printHeader <B>PrintHeader</B>},
   * {@link #printAllFields <B>printAllFields</B>} and
   * {@link #printAllMethods <B>printAllMethods</B>}
   **/
 public void printOutputClass()
  {
  printHeader();
  psOutputFile.println( "\n {" );
  printAllFields(); 
  printConstructor();
  printAllMethods(); 
  psOutputFile.println( " }" );
  }

  /**
   * This method print the specified field (whith its modifiers and type) to the output file.
   * @param field field to be printed.
   **/
 public void printField( OJField field )
  {
  OJModifier ojmModifiers = field.getModifiers();
  psOutputFile.print( " " + ojmModifiers );
  psOutputFile.print( " " + field.getType().getName() );
  psOutputFile.println( " " + field.getName() + ";" );
  }

  /**
   * This method print all fields calling 
   * {@link #printField <B>printField</B>} method for each field.
   **/
 public void printAllFields()
  {
  for( int i=0 ; i < ojfMyFields.size() ; i++ )
    printField( (OJField)ojfMyFields.elementAt(i) );
  }

  /**
   * This method print the constructor of the class to the output file (with the
   * exception it throws).
   * In the output class the constructor simply consist of a call to
   * <code>super()</code> with the given parameters.
   **/
 public void printConstructor()
  {
  for( int j=0; j<constructors.size(); j++ )
    {
    OJMethod constr = (OJMethod)constructors.elementAt(j);

    psOutputFile.print( "\n public " + className + "( " );

    try{
      OJClass[] ojTypes = constr.getParameterTypes();
      String[] strNames = constr.getParameters();
      for( int i=0; i<ojTypes.length; )
        {
        psOutputFile.print( ojTypes[i].getName() + " " + strNames[i] );
        if( ++i<ojTypes.length )
          psOutputFile.print( ", " );
        }
      psOutputFile.print( " )" );

      OJClass[] ojException = constr.getExceptionTypes();
      if( ojException.length > 0 )
        psOutputFile.print( " throws ");
      for( int i=0; i<ojException.length ;  )
        {
        psOutputFile.print( ojException[i].getName() );
        if( ++i < ojException.length )
          psOutputFile.print( ", " );
        }  
      psOutputFile.println( "\n  {" );

      psOutputFile.println( "  " + fstLineConstr.get(j) );
      StatementList slBody = constr.getBody();
      for(int i=0; i<slBody.size(); i++)
        {
        psOutputFile.println( "  " + slBody.get(i) );
        }

      psOutputFile.println( "  }" );
      }
    catch( CannotAlterException e )
      {
      PreUtils.printExceptionText( "FATAL ERROR: Unable to analized the constructor", e );
      System.exit(-1);
      }
    }
  }

  /**
   * This method print the specified method to the output file.
   * @param method method to print.
   **/
 public void printMethod( OJMethod method )
  {
  try{
    OJModifier ojmModifiers = method.getModifiers();
    psOutputFile.print( "\n " + ojmModifiers );
    psOutputFile.print( " " + method.getReturnType().getName() );
    psOutputFile.print( " " + method.getName() + "( " );
    String[] paramNames = method.getParameters();
    OJClass[] paramTypes = method.getParameterTypes();

    for(int i=0; i<paramNames.length; )
      {
      psOutputFile.print( paramTypes[i].getName() + " " + paramNames[i] );
      if( ++i < paramNames.length )
	psOutputFile.print( ", " );
      }
    psOutputFile.print( " )" );
    OJClass[] excTypes = method.getExceptionTypes();
    if( excTypes.length>0 )
      {
      psOutputFile.print( " throws " );
      for(int i=0; i<excTypes.length; )
	{
	psOutputFile.print( excTypes[i].getName() );
	if( ++i < excTypes.length )
	  psOutputFile.print( ", " );
	}
      }
    psOutputFile.println( "\n  {" );
    StatementList slBody = method.getBody();
    for(int i=0; i<slBody.size(); i++)
      {
      psOutputFile.println( "  " + slBody.get(i) );
      }
    psOutputFile.println( "  }\n" );
    }
  catch( CannotAlterException e )
    {
    PreUtils.printExceptionText( "Unable to create method ", e );
    }
  }

  /**
   * This method print all methods to the output file calling
   * {@link #printMethod <B>printMethod</B>} method for each OJMethod.
   **/
 public void printAllMethods()
  {
  for( int i=0 ; i < ojmMyMethods.size() ; i++ )
    printMethod( (OJMethod)ojmMyMethods.elementAt(i) );
  }

 }
