﻿module OO

//////////////////////
// record of lambdas
//////////////////////
type RecCounter = { incr : Unit -> Unit; curr : Unit -> int }

let mk_counter () =
  let i = ref 0
  let incr () = i := !i + 1
  let curr () = !i
  in { incr = incr; curr = curr }

let print_counter () =
  let i = ref 0
  let incr () = 
    do printf "incremented from %d to %d\n" (!i) (!i + 1)
    i := !i + 1
  let curr () = !i
  in { incr = incr; curr = curr }

let counter_from z =
  let i = ref z
  let incr () = 
    i := !i + 2
  let curr () = !i
  in { incr = incr; curr = curr }

let c1 = mk_counter()
let c2 = print_counter()
let c3 = counter_from 10

let use_counter c =
  for i = 0 to 10 do
    do c.incr()
    do printf "%d\n" (c.curr())

do use_counter c1
do use_counter c2
do use_counter c3



//////////////////////
// C# -style objects
//////////////////////
type ICounter =
  abstract member Incr : Unit -> Unit
  abstract member Curr : int with get

type Counter() =
  let mutable i = 0
  interface ICounter with
    member this.Incr() = i <- i + 1
    member this.Curr with get() = i

type PrintingCounter() =
  let mutable i = 0
  interface ICounter with
    member this.Incr() = 
      do printf "incremented from %d to %d\n" i (i + 1)
      i <- i + 1
    member this.Curr with get() = i

type CounterFrom(z) =
  let mutable i = z
  interface ICounter with
    member this.Incr() = i <- i + 1
    member this.Curr with get() = i
  
let c1' = new Counter()
let c2' = new PrintingCounter()
let c3' = new CounterFrom(10)

let use_counter' (c:#ICounter) =
  for i = 0 to 10 do
    do c.Incr()
    do printf "%d\n" c.Curr

do use_counter' c1'
do use_counter' c2'
do use_counter' c3'


//////////////////////
// object expressions
//////////////////////
let c4' = 
  let i = ref 0
  { new ICounter with
      member this.Incr() = i := !i + 2
      member this.Curr with get() = !i }
  
do use_counter' c4'
